/** fMSX: portable MSX emulator ******************************/
/**                                                         **/
/**                          Disk.c                         **/
/**                                                         **/
/** This file contains standard disk access drivers working **/
/** with disk images from files.                            **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1994-2005                 **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/
#include "MSX.h"
#include "Floppy.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
//#include <fat.h>

#ifdef MSDOS
#include <io.h>
#endif

#ifdef WINDOWS
#include <io.h>
#endif

#ifdef UNIX
//#include <unistd.h>
#endif

#ifndef O_BINARY
#define O_BINARY 0
#endif

extern void LOG( char * str , ... );


static FILE* Drives[2] = { NULL,NULL }; /* Disk image files */
static byte *Images[2] = { 0,0 };           /* Disk image buffers */
static int RdOnly[2];                       /* 1 = read-only      */ 

/** DiskPresent() ********************************************/
/** Return 1 if disk drive with a given ID is present.      **/
/*************************************************************/
int DiskPresent(int ID)
{
  return (( (ID>=0)&&(ID<MAXDRIVES)&&((Drives[ID]!=NULL)||Images[ID]) )?1:0);
}

/** DiskRead() ***********************************************/
/** Read requested sector from the drive into a buffer.     **/
/*************************************************************/
int DiskRead(int ID,byte *Buf,int N)
{
//  LOG( "DiskRead 1" );
  if((ID>=0)&&(ID<MAXDRIVES))
  {
//  LOG( "DiskRead 2" );
    if(Images[ID]&&(N>=0)&&(N<DSK_SECS_PER_DISK))
    { memcpy(Buf,Images[ID]+N*512,512);return(1); }
//  LOG( "DiskRead 3" );

    if(Drives[ID]!=NULL){
      if(fseek(Drives[ID],N*512L,SEEK_SET)>=0){
//		LOG( "DiskRead 4" );
		if( fread(Buf,1,512,Drives[ID])==512 ){
//		LOG( "DiskRead 5" );
	        return(true);
		}else{
//		LOG( "DiskRead 6" );
			return(false);
		}
      }
    }
  }
//  LOG( "DiskRead 7" );
  return(0);
}

/** DiskWrite() **********************************************/
/** Write contents of the buffer into a given sector of the **/
/** disk.                                                   **/
/*************************************************************/
int DiskWrite(int ID,byte *Buf,int N)
{
  if((ID>=0)&&(ID<MAXDRIVES)&&!RdOnly[ID])
  {
    if(Images[ID]&&(N>=0)&&(N<DSK_SECS_PER_DISK))
    { memcpy(Images[ID]+N*512,Buf,512);return(1); }

    if(Drives[ID]!=NULL)
      if(fseek(Drives[ID],N*512L,SEEK_SET)>=0)
        return((fwrite(Buf,1,512,Drives[ID])==512)?1:0);
  }
  return(0);
}

/** ChangeDisk() *********************************************/
/** Change disk image in a given drive. Closes current disk **/
/** image if Name=0 was given. Returns 1 on success or 0 on **/
/** failure.                                                **/
/*************************************************************/
byte ChangeDisk(byte ID,char *Name)
{

  byte Magic[512];
  int ret;

  if(Name){
    LOG( "[ChgDsk]ID=%d NAME=%s",ID,Name );
  }
  /* We only have MAXDRIVES drives */
  if(ID>=MAXDRIVES) return(0);

//  LOG( "ChangeDisk 2" );

  /* Close previous disk image */
  if(Drives[ID]!=NULL) { fclose(Drives[ID]);Drives[ID]=NULL; }
  if(Images[ID])    { free(Images[ID]);Images[ID]=0; }
//  LOG( "ChangeDisk 3" );

  /* If no disk image given, consider drive empty */
  if(!Name) return(1);
  if( !*Name ) return(1);

  //  LOG( "ChangeDisk 4" );

  /* Open new disk image */
  Drives[ID]=fopen(Name,"r+");
  RdOnly[ID]=0;

//    LOG( "ChangeDisk 5" );

  /* If failed to open for writing, open read-only */
  if(Drives[ID]==NULL)
  {
//    LOG( "ChangeDisk 6" );
    Drives[ID]=fopen(Name,"r");
    RdOnly[ID]=1;
  }
//  LOG( "ChangeDisk 7" );
  /* Verify that open file is really a disk image */
  if(Drives[ID]!=NULL){
    LOG( "ChangeDisk 8" );
  	ret = fread(Magic,512,1,Drives[ID]);
  	if((ret!=1)||(Magic[0]!=0xEB))
    { 
      LOG( "ChangeDisk ERR ret=%d Mgc[0]=%x",ret,Magic[0] );
      fclose(Drives[ID]);RdOnly[ID]=0;Drives[ID]=NULL; 
    }
  }

#if 0
  /* If failed to open as a plain file, */
  /* try creating disk image in memory  */
  if(Drives[ID]<0) Images[ID]=DSKLoad(Name,0);
#endif

  /* Return operation result */
  return(Images[ID]||(Drives[ID]!=NULL));
}
