/** fMSX: portable MSX emulator ******************************/
/**                                                         **/
/**                         Common.h                        **/
/**                                                         **/
/** This file contains standard screen refresh drivers      **/
/** common for X11, VGA, and other "chunky" bitmapped video **/
/** implementations. It also includes dummy sound drivers   **/
/** for fMSX.                                               **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1994-2005                 **/
/**               John Stiles     1996                      **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/

#include "../CE/CE_Driver.h"
#include "Common.h"

static int FirstLine;     /* First scanline in the XBuf */

/** ClearLine() **********************************************/
/** Clear pixels from P with color C.                       **/
/*************************************************************/
static void ClearLine( pixel *P, pixel C, int w)
{
  register int J;
  for(J=0;J<w;J++) P[J]=C;
//  P+=544;
//  for(J=0;J<w;J++) P[J]=C;
}

/** RefreshBorder() ******************************************/
/** This function is called from RefreshLine#() to refresh  **/
/** the screen border. It returns a pointer to the start of **/
/** scanline Y in XBuf or 0 if scanline is beyond XBuf.     **/
/*************************************************************/
static pixel * RefreshBorder(byte Y, pixel C)
{
  register pixel *P;
  register int H;
  register int sy;

  if( !Usex2Mode ){
	  /* First line number in the buffer */
	  if(!Y) FirstLine=(ScanLines212? 8:18)+VAdjust;

	  /* Return 0 if we've run out of the screen buffer due to overscan */
	  if(Y+FirstLine>=HEIGHT) return(0);

	  /* Set up the transparent color */
	  XPal[0]=(!BGColor||SolidColor0)? XPal0:XPal[BGColor];

	  /* Start of the buffer */
	  P=(pixel *)XBuf;

	  /* Paint top of the screen */
	  if(!Y) for(H=WIDTH*FirstLine-1;H>=0;H--) P[H]=C;

	  /* Start of the line */
	  P+=WIDTH*(FirstLine+Y);

	  /* Paint left/right borders */
	  for(H=8+HAdjust;H>0;H--) P[H-1]=C;
	  for(H=8-HAdjust;H>0;H--) P[272-H]=C;

	  /* Paint bottom of the screen */
	  H=ScanLines212? 212:192;
	  if(Y==H-1) for(H=WIDTH*(HEIGHT-H-FirstLine+1)-1;H>=WIDTH;H--) P[H]=C;

	  /* Return pointer to the scanline in XBuf */
	  return(P+8+HAdjust);
  }else{
	  sy = Y * 2;
	  /* First line number in the buffer */
	  if(!sy) FirstLine=(ScanLines212? 16:36)+VAdjust*2;

	  /* Return 0 if we've run out of the screen buffer due to overscan */
	  if(sy+FirstLine>=HEIGHT) return(0);

	  /* Set up the transparent color */
	  XPal[0]=(!BGColor||SolidColor0)? XPal0:XPal[BGColor];

	  /* Start of the buffer */
	  P=(pixel *)XBuf;

	  /* Paint top of the screen */
	  if(!sy) for(H=WIDTH*FirstLine-1;H>=0;H--) P[H]=C;

	  /* Start of the line */
	  P+=WIDTH*(FirstLine+sy);

	  /* Paint left/right borders */
	  for(H=16+HAdjust*2;H>0;H--){
		P[H-1]=C;
		if( !UseInterless ) P[H-1+WIDTH]=C;
	  }
	  for(H=16-HAdjust*2;H>0;H--){
		P[544-H]=C;
		if( !UseInterless ) P[544-H+WIDTH]=C;
	  }

	  /* Paint bottom of the screen */
	  H=ScanLines212? 424:384;
	  if(sy==H-2) for(H=WIDTH*(HEIGHT-H-FirstLine+1)-1;H>=WIDTH;H--) P[H]=C;

	  /* Return pointer to the scanline in XBuf */
	  return(P+16+HAdjust*2);

  }
}

/** RefreshLineF() *******************************************/
/** Dummy refresh function called for non-existing screens. **/
/*************************************************************/
static void RefreshLineF(register byte Y)
{
  register pixel *P;

  if(Verbose>1)
    LOG
    (
      "ScrMODE %d: ChrTab=%X ChrGen=%X ColTab=%X SprTab=%X SprGen=%X\n",
      ScrMode,ChrTab-VRAM,ChrGen-VRAM,ColTab-VRAM,SprTab-VRAM,SprGen-VRAM
    );

  P=RefreshBorder(Y,XPal[BGColor]);
  if(P) ClearLine(P,XPal[BGColor],512);
}

/** RefreshLine0() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN0.                 **/
/*************************************************************/
static void RefreshLine0(register byte Y)
{
  register pixel *P,*OP,FC,BC;
  register byte X,*T,*G;
  int i;
  BC=XPal[BGColor];
  OP = P =RefreshBorder(Y,BC);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BC,512);
  else
  {
    P[0]=P[1]=P[2]=P[3]=P[4]=P[5]=P[6]=P[7]=P[8]=BC;
    P[9]=P[10]=P[11]=P[12]=P[13]=P[14]=P[15]=P[16]=P[17]=BC;
    G=ChrGen+((Y+VScroll)&0x07);
    T=ChrTab+40*(Y>>3);
    FC=XPal[FGColor];
    P+=18;

    for(X=0;X<40;X++,T++,P+=12)
    {
      Y=G[(int)*T<<3];
      P[ 0]=P[ 1]=Y&0x80? FC:BC;
	  P[ 2]=P[ 3]=Y&0x40? FC:BC;
      P[ 4]=P[ 5]=Y&0x20? FC:BC;
	  P[ 6]=P[ 7]=Y&0x10? FC:BC;
      P[ 8]=P[ 9]=Y&0x08? FC:BC;
	  P[10]=P[11]=Y&0x04? FC:BC;
    }
    P[0]=P[1]=P[2]=P[3]=P[4]=P[5]=P[6]=BC;
    P[7]=P[8]=P[9]=P[10]=P[11]=P[12]=P[13]=BC;

	if( !UseInterless ) {
		memcpy( OP+WIDTH, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine1() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN1, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
static void RefreshLine1(register byte Y)
{
  register pixel *P,*OP,FC,BC;
  register byte K,X,*T,*G,*R,C;
  register int cnt;
  byte ZBuf[304];

  OP = P=RefreshBorder(Y,XPal[BGColor]);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    Sprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    G=ChrGen+(Y&0x07);
    T=ChrTab+((int)(Y&0xF8)<<2);

    for(X=0;X<32;X++,T++,P+=16,R+=8)
    {
      K=ColTab[*T>>3];
      FC=XPal[K>>4];
      BC=XPal[K&0x0F];
      K=G[(int)*T<<3];

	  C=R[0];P[ 0]=P[ 1]=C? XPal[C]:(K&0x80)? FC:BC;
	  C=R[1];P[ 2]=P[ 3]=C? XPal[C]:(K&0x40)? FC:BC;
	  C=R[2];P[ 4]=P[ 5]=C? XPal[C]:(K&0x20)? FC:BC;
	  C=R[3];P[ 6]=P[ 7]=C? XPal[C]:(K&0x10)? FC:BC;
	  C=R[4];P[ 8]=P[ 9]=C? XPal[C]:(K&0x08)? FC:BC;
	  C=R[5];P[10]=P[11]=C? XPal[C]:(K&0x04)? FC:BC;
	  C=R[6];P[12]=P[13]=C? XPal[C]:(K&0x02)? FC:BC;
	  C=R[7];P[14]=P[15]=C? XPal[C]:(K&0x01)? FC:BC;
    }
	if( !UseInterless ) {
		memcpy( OP+WIDTH, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine2() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN2, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
static void RefreshLine2(register byte Y)
{
  register pixel *P,*OP,FC,BC;
  register byte K,X,*T,*R,C;
  register int I,J;
  byte ZBuf[304];
  int cnt;

  OP=P=RefreshBorder(Y,XPal[BGColor]);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    Sprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    T=ChrTab+((int)(Y&0xF8)<<2);
    I=((int)(Y&0xC0)<<5)+(Y&0x07);

    for(X=0;X<32;X++,T++,P+=16,R+=8)
    {
		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];
		C=R[0];P[ 0]=P[ 1]=C? XPal[C]:(K&0x80)? FC:BC;
		C=R[1];P[ 2]=P[ 3]=C? XPal[C]:(K&0x40)? FC:BC;
		C=R[2];P[ 4]=P[ 5]=C? XPal[C]:(K&0x20)? FC:BC;
		C=R[3];P[ 6]=P[ 7]=C? XPal[C]:(K&0x10)? FC:BC;
		C=R[4];P[ 8]=P[ 9]=C? XPal[C]:(K&0x08)? FC:BC;
		C=R[5];P[10]=P[11]=C? XPal[C]:(K&0x04)? FC:BC;
		C=R[6];P[12]=P[13]=C? XPal[C]:(K&0x02)? FC:BC;
		C=R[7];P[14]=P[15]=C? XPal[C]:(K&0x01)? FC:BC;
    }
	if( !UseInterless ) {
		memcpy( OP+WIDTH, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine3() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN3, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
static void RefreshLine3(register byte Y)
{
  register pixel *P,*OP;
  register byte X,K,*T,*G,*R,C;
  byte ZBuf[304];
  int cnt;

  OP=P=RefreshBorder(Y,XPal[BGColor]);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    Sprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    T=ChrTab+((int)(Y&0xF8)<<2);
    G=ChrGen+((Y&0x1C)>>2);

    for(X=0;X<32;X++,T++,P+=16)
    {
      K=G[(int)*T<<3];
	  C=R[0];P[ 0]=P[ 1]=XPal[C?C:(K>>4  )];
	  C=R[1];P[ 2]=P[ 3]=XPal[C?C:(K>>4  )];
	  C=R[2];P[ 4]=P[ 5]=XPal[C?C:(K>>4  )];
	  C=R[3];P[ 6]=P[ 7]=XPal[C?C:(K>>4  )];
	  C=R[4];P[ 8]=P[ 9]=XPal[C?C:(K&0x0F)];
	  C=R[5];P[10]=P[11]=XPal[C?C:(K&0x0F)];
	  C=R[6];P[12]=P[13]=XPal[C?C:(K&0x0F)];
	  C=R[7];P[14]=P[15]=XPal[C?C:(K&0x0F)];
    }
	if( !UseInterless ) {
		memcpy( OP+WIDTH, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine4() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN4, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
static void RefreshLine4(register byte Y)
{
  register pixel *P,*OP,FC,BC;
  register byte K,X,C,*T,*R,XE;
  register int I,J,cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,XPal[BGColor]);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    T=ChrTab+((int)(Y&0xF8)<<2);
    I=((int)(Y&0xC0)<<5)+(Y&0x07);

    for(X=0;X<32;X++,R+=8,P+=16,T++)
    {
      J=(int)*T<<3;
      K=ColTab[(I+J)&ColTabM];
      FC=XPal[K>>4];
      BC=XPal[K&0x0F];
      K=ChrGen[(I+J)&ChrGenM];

      C=R[0];P[ 0]=P[ 1]=C? XPal[C]:(K&0x80)? FC:BC;
      C=R[1];P[ 2]=P[ 3]=C? XPal[C]:(K&0x40)? FC:BC;
      C=R[2];P[ 4]=P[ 5]=C? XPal[C]:(K&0x20)? FC:BC;
      C=R[3];P[ 6]=P[ 7]=C? XPal[C]:(K&0x10)? FC:BC;
      C=R[4];P[ 8]=P[ 9]=C? XPal[C]:(K&0x08)? FC:BC;
      C=R[5];P[10]=P[11]=C? XPal[C]:(K&0x04)? FC:BC;
      C=R[6];P[12]=P[13]=C? XPal[C]:(K&0x02)? FC:BC;
      C=R[7];P[14]=P[15]=C? XPal[C]:(K&0x01)? FC:BC;
    }
    
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine5() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN5, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
static void RefreshLine5(register byte Y)
{
  register pixel *P,*OP;
  register byte I,X,*T,*R;
  byte ZBuf[304];
  int cnt;

  OP=P=RefreshBorder(Y,XPal[BGColor]);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<7)&ChrTabM&0x7FFF);

    for(X=0;X<32;X++,R+=8,P+=16,T+=4)
    {
      I=R[ 0];P[ 0]=P[ 1]=XPal[I? I:T[0]>>4];
      I=R[ 1];P[ 2]=P[ 3]=XPal[I? I:T[0]&0x0F];
      I=R[ 2];P[ 4]=P[ 5]=XPal[I? I:T[1]>>4];
      I=R[ 3];P[ 6]=P[ 7]=XPal[I? I:T[1]&0x0F];
      I=R[ 4];P[ 8]=P[ 9]=XPal[I? I:T[2]>>4];
      I=R[ 5];P[10]=P[11]=XPal[I? I:T[2]&0x0F];
      I=R[ 6];P[12]=P[13]=XPal[I? I:T[3]>>4];
      I=R[ 7];P[14]=P[15]=XPal[I? I:T[3]&0x0F];
    }
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine6() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN6, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
static void RefreshLine6(register byte Y)
{
  register pixel *P,*OP;
  register byte X,*T,*R,C;
  register int cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,XPal[BGColor&0x03]);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor&0x03],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<7)&ChrTabM&0x7FFF);

    for(X=0;X<32;X++)
    {
      C=R[0];P[ 0]=XPal[C? C:(T[0]>>6)];
      C=R[0];P[ 1]=XPal[C? C:(T[0]>>4)&0x03];
      C=R[1];P[ 2]=XPal[C? C:(T[0]>>2)&0x03];
      C=R[1];P[ 3]=XPal[C? C: T[0]    &0x03];
      C=R[2];P[ 4]=XPal[C? C:(T[1]>>6)];
      C=R[2];P[ 5]=XPal[C? C:(T[1]>>4)&0x03];
      C=R[3];P[ 6]=XPal[C? C:(T[1]>>2)&0x03];
      C=R[3];P[ 7]=XPal[C? C: T[1]    &0x03];
      C=R[4];P[ 8]=XPal[C? C:(T[2]>>6)];
      C=R[4];P[ 9]=XPal[C? C:(T[2]>>4)&0x03];
      C=R[5];P[10]=XPal[C? C:(T[2]>>2)&0x03];
      C=R[5];P[11]=XPal[C? C: T[2]    &0x03];
      C=R[6];P[12]=XPal[C? C:(T[3]>>6)];
      C=R[6];P[13]=XPal[C? C:(T[3]>>4)&0x03];
      C=R[7];P[14]=XPal[C? C:(T[3]>>2)&0x03];
      C=R[7];P[15]=XPal[C? C: T[3]    &0x03];
      R+=8;P+=16;T+=4;
    }
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}
  
/** RefreshLine7() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN7, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
static void RefreshLine7(register byte Y)
{
  register pixel *P,*OP;
  register byte C,X,*T,*R;
  register int cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,XPal[BGColor]);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

    for(X=0;X<32;X++)
    {
      C=R[0];P[ 0]=XPal[C? C:T[0]>>4];
      C=R[0];P[ 1]=XPal[C? C:T[0]   &0xF];
      C=R[1];P[ 2]=XPal[C? C:T[1]>>4];
      C=R[1];P[ 3]=XPal[C? C:T[1]   &0xF];
      C=R[2];P[ 4]=XPal[C? C:T[2]>>4];
      C=R[2];P[ 5]=XPal[C? C:T[2]   &0xF];
      C=R[3];P[ 6]=XPal[C? C:T[3]>>4];
      C=R[3];P[ 7]=XPal[C? C:T[3]   &0xF];
      C=R[4];P[ 8]=XPal[C? C:T[4]>>4];
      C=R[4];P[ 9]=XPal[C? C:T[4]   &0xF];
      C=R[5];P[10]=XPal[C? C:T[5]>>4];
      C=R[5];P[11]=XPal[C? C:T[5]   &0xF];
      C=R[6];P[12]=XPal[C? C:T[6]>>4];
      C=R[6];P[13]=XPal[C? C:T[6]   &0xF];
      C=R[7];P[14]=XPal[C? C:T[7]>>4];
      C=R[7];P[15]=XPal[C? C:T[7]   &0xF];
      R+=8;P+=16;T+=8;
    }
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine8() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN8, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
static void RefreshLine8(register byte Y)
{
  static byte SprToScr[16] =
  {
    0x00,0x02,0x10,0x12,0x80,0x82,0x90,0x92,
    0x49,0x4B,0x59,0x5B,0xC9,0xCB,0xD9,0xDB
  };
  register pixel *P,*OP;
  register byte C,X,*T,*R;
  register int cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,BPal[VDP[7]]);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BPal[VDP[7]],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

    for(X=0;X<32;X++,T+=8,R+=8,P+=16)
    {
      C=R[0];P[ 0]=P[ 1]=BPal[C? SprToScr[C]:T[0]];
      C=R[1];P[ 2]=P[ 3]=BPal[C? SprToScr[C]:T[1]];
      C=R[2];P[ 4]=P[ 5]=BPal[C? SprToScr[C]:T[2]];
      C=R[3];P[ 6]=P[ 7]=BPal[C? SprToScr[C]:T[3]];
      C=R[4];P[ 8]=P[ 9]=BPal[C? SprToScr[C]:T[4]];
      C=R[5];P[10]=P[11]=BPal[C? SprToScr[C]:T[5]];
      C=R[6];P[12]=P[13]=BPal[C? SprToScr[C]:T[6]];
      C=R[7];P[14]=P[15]=BPal[C? SprToScr[C]:T[7]];
    }
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine10() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN10/11, including   **/
/** sprites in this line.                                   **/
/*************************************************************/
static void RefreshLine10(register byte Y)
{
  register pixel *P,*OP;
  register byte C,X,*T,*R;
  register int J,K,cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,BPal[VDP[7]]);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BPal[VDP[7]],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

	/* Draw first 4 pixels */
	C=R[0];P[ 0]=P[ 1]=C? XPal[C]:BPal[VDP[7]];
	C=R[1];P[ 2]=P[ 3]=C? XPal[C]:BPal[VDP[7]];
	C=R[2];P[ 4]=P[ 5]=C? XPal[C]:BPal[VDP[7]];
	C=R[3];P[ 6]=P[ 7]=C? XPal[C]:BPal[VDP[7]];
	R+=4;P+=8;

    for(X=0;X<63;X++,T+=4,R+=4,P+=8)
    {
      K=(T[0]&0x07)|((T[1]&0x07)<<3);
      if(K&0x20) K-=64;
      J=(T[2]&0x07)|((T[3]&0x07)<<3);
      if(J&0x20) J-=64;

      C=R[0];Y=T[0]>>3;P[ 0]=P[ 1]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
      C=R[1];Y=T[1]>>3;P[ 2]=P[ 3]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
      C=R[2];Y=T[2]>>3;P[ 4]=P[ 5]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
      C=R[3];Y=T[3]>>3;P[ 6]=P[ 7]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
    }

	if( !UseInterless ) {
		memcpy( OP+WIDTH, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine12() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN12, including      **/
/** sprites in this line.                                   **/
/*************************************************************/
static void RefreshLine12(register byte Y)
{
  register pixel *P,*OP;
  register byte C,X,*T,*R;
  register int J,K,cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,BPal[VDP[7]]);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BPal[VDP[7]],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

    if(HScroll512&&(HScroll>255)) T=(byte *)((int)T^0x10000);
    T+=HScroll&0xFC;

    /* Draw first 4 pixels */
    C=R[0];P[ 0]=P[ 1]=C? XPal[C]:BPal[VDP[7]];
    C=R[1];P[ 2]=P[ 3]=C? XPal[C]:BPal[VDP[7]];
    C=R[2];P[ 4]=P[ 5]=C? XPal[C]:BPal[VDP[7]];
    C=R[3];P[ 6]=P[ 7]=C? XPal[C]:BPal[VDP[7]];
    R+=4;P+=8;

    for(X=1;X<63;X++,T+=4,R+=4,P+=8)
    {
      K=(T[0]&0x07)|((T[1]&0x07)<<3);
      if(K&0x20) K-=64;
      J=(T[2]&0x07)|((T[3]&0x07)<<3);
      if(J&0x20) J-=64;

      C=R[0];P[ 0]=P[ 1]=C? XPal[C]:YJKColor(T[0]>>3,J,K);
      C=R[1];P[ 2]=P[ 3]=C? XPal[C]:YJKColor(T[1]>>3,J,K);
      C=R[2];P[ 4]=P[ 5]=C? XPal[C]:YJKColor(T[2]>>3,J,K);
      C=R[3];P[ 6]=P[ 7]=C? XPal[C]:YJKColor(T[3]>>3,J,K);
    }
	if( !UseInterless ) {
		memcpy( OP+WIDTH, OP, sizeof(pixel) * 512 ); 
	}
  }
}


/** RefreshLineTx80() ****************************************/
/** Refresh line Y (0..191/211) of TEXT80.                  **/
/*************************************************************/
static void RefreshLineTx80(register byte Y)
{
  register pixel *P,*OP,FC,BC;
  register byte X,M,*T,*C,*G;

  BC=XPal[BGColor];
  OP=P=RefreshBorder(Y,BC);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BC,512);
  else
  {
    P[0]=P[1]=P[2]=P[3]=P[4]=P[5]=P[6]=P[7]=P[8]=BC;
    P[9]=P[10]=P[11]=P[12]=P[13]=P[14]=P[15]=P[16]=P[17]=BC;
    G=ChrGen+((Y+VScroll)&0x07);
    T=ChrTab+80*(Y>>3);
    C=ColTab+10*(Y>>3);
    P+=18;

    for(X=0,M=0x00;X<80;X++,T++,P+=6)
    {
      if(!(X&0x07)) M=*C++;
      if(M&0x80) { FC=XPal[XFGColor];BC=XPal[XBGColor]; }
      else       { FC=XPal[ FGColor];BC=XPal[BGColor]; }
      M<<=1;
      Y=*(G+((int)*T<<3));
      P[0]=Y&0x80? FC:BC;
      P[1]=Y&0x40? FC:BC;
      P[2]=Y&0x20? FC:BC;
      P[3]=Y&0x10? FC:BC;
      P[4]=Y&0x08? FC:BC;
      P[5]=Y&0x04? FC:BC;
    }

    P[0]=P[1]=P[2]=P[3]=P[4]=P[5]=P[6]=BC;
    P[7]=P[8]=P[9]=P[10]=P[11]=P[12]=P[13]=BC;

	if( !UseInterless ) {
		memcpy( OP+WIDTH, OP, sizeof(pixel) * 512 ); 
	}
  }
}

void (*RefreshLineList3x2[MAXSCREEN+2])(byte Y) = 
{
  RefreshLine0,   /* SCR 0:  TEXT 40x24  */
  RefreshLine1,   /* SCR 1:  TEXT 32x24  */
  RefreshLine2,   /* SCR 2:  BLK 256x192 */
  RefreshLine3,   /* SCR 3:  64x48x16    */
  RefreshLine4,   /* SCR 4:  BLK 256x192 */
  RefreshLine5,   /* SCR 5:  256x192x16  */
  RefreshLine6,   /* SCR 6:  512x192x4   */
  RefreshLine7,   /* SCR 7:  512x192x16  */
  RefreshLine8,   /* SCR 8:  256x192x256 */
  RefreshLineF,   /* SCR 9:  NONE        */
  RefreshLine10,  /* SCR 10: YAE 256x192 */
  RefreshLine10,  /* SCR 11: YAE 256x192 */
  RefreshLine12,  /* SCR 12: YJK 256x192 */
  RefreshLineTx80 /* SCR 0:  TEXT 80x24  */
};
