/** fMSX: portable MSX emulator ******************************/
/**                                                         **/
/**                         Common.h                        **/
/**                                                         **/
/** This file contains standard screen refresh drivers      **/
/** common for X11, VGA, and other "chunky" bitmapped video **/
/** implementations. It also includes dummy sound drivers   **/
/** for fMSX.                                               **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1994-2005                 **/
/**               John Stiles     1996                      **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/

/** INLINE ***************************************************/
/** Different compilers inline C functions differently.     **/
/*************************************************************/
#include "../CE/CE_Driver.h"
#include "Common.h"

#ifdef __GNUC__
#define INLINE inline
#else

#define INLINE
#define register
#endif

static int FirstLine = 18;     /* First scanline in the XBuf */

static void  Sprites(byte Y,pixel *Line);
static void  ColorSprites(byte Y,byte *ZBuf);
//static pixel *RefreshBorder(byte Y,pixel C);
static pixel * RefreshBorder(byte y, pixel c, int w1 , int w2);
static void ClearLine( pixel *P, pixel C, int w);
static pixel YJKColor(int Y,int J,int K);

/** RefreshScreen() ******************************************/
/** Refresh screen. This function is called in the end of   **/
/** refresh cycle to show the entire screen.                **/
/*************************************************************/
//void RefreshScreen(void) { PutImage(); }

/** ClearLine() **********************************************/
/** Clear 256 pixels from P with color C.                   **/
/*************************************************************/
static void ClearLine( pixel *P, pixel C, int w)
{
  register int J;
  for(J=0;J<w;J++) P[J]=C;
  if( Usex2Mode && !UseInterless ){
	  P+=544;
	  for(J=0;J<w;J++) P[J]=C;
  }	
}
/** YJKColor() ***********************************************/
/** Given a color in YJK format, return the corresponding   **/
/** palette entry.                                          **/
/*************************************************************/
static INLINE pixel YJKColor(register int Y,register int J,register int K)
{
  register int R,G,B;
		
  R=Y+J;
  G=Y+K;
  B=((5*Y-2*J-K)/4);

  R=R<0? 0:R>31? 31:R;
  G=G<0? 0:G>31? 31:G;
  B=B<0? 0:B>31? 31:B;

  return(BPal[(R&0x1C)|((G&0x1C)<<3)|(B>>3)]);
}

/** RefreshBorder() ******************************************/
/** This function is called from RefreshLine#() to refresh  **/
/** the screen border. It returns a pointer to the start of **/
/** scanline Y in XBuf or 0 if scanline is beyond XBuf.     **/
/*************************************************************/
pixel * RefreshBorder(byte y, pixel c, int w1 , int w2)
{
//  register pixel *P;
//  register int H;

	pixel *P, *OP;
	int i,j,H;
    int bsiz;


	/* First line number in the buffer */
	if(y==0) FirstLine = (ScanLines212? 8:18) + VAdjust;

	/* Return 0 if we've run out of the screen buffer due to overscan */
	if(y+FirstLine>=HEIGHT) return(0);

	/* Set up the transparent color */
	XPal[0]=(!BGColor||SolidColor0)? XPal0:XPal[BGColor];

	if( !UseDrawBorder ){
		if( Usex2Mode ){
			return((pixel *)XBuf+WIDTH*(FirstLine+y*2)+HAdjust);
		}else{
			return((pixel *)XBuf+WIDTH*(FirstLine+y  )+HAdjust);
		}
	}


	/* Start of the buffer */
	P=(pixel *)XBuf;
	
	/* Paint top of the screen */
	if(!y){
		if( Usex2Mode ){
			if( !UseInterless ){
				OP = P + 544;
				for(i=(FirstLine-1)*2;i>=0;i-=2){
					for(H=w2-1;H>=0;H--) OP[WIDTH*i+H] = P[WIDTH*i+H]=c;
				}
			}else{
				for(i=(FirstLine-1)*2;i>=0;i-=2){
					for(H=w2-1;H>=0;H--) P[WIDTH*i+H]=c;
				}
			}
		}else{
			for(i=FirstLine-1;i>=0;i--){
				for(H=w2-1;H>=0;H--) P[WIDTH*i+H]=c;
			}
		}
	}

	/* Start of the line */
	if( Usex2Mode ){
		P+=WIDTH*(FirstLine+y)*2;
		OP = P + WIDTH;
	}else{
		P+=WIDTH*(FirstLine+y);
	}
	/* Paint left/right borders */
	bsiz = (w2-w1)/2;
	if( Usex2Mode && !UseInterless ){
		for(H=bsiz+HAdjust;H>0;H--) OP[H-1] = P[H-1]=c;
		for(H=bsiz-HAdjust+1;H>0;H--) OP[w2-H]=P[w2-H]=c;
	}else{
		for(H=bsiz+HAdjust;H>0;H--) P[H-1]=c;
		for(H=bsiz-HAdjust+1;H>0;H--) P[w2-H]=c;
	}
	/* Paint bottom of the screen */
	H=ScanLines212? 212:192;
	if(y==H-1){
		if( Usex2Mode ){
			if( !UseInterless ){
				for(i=(HEIGHT-H-FirstLine)*2;i>=0;i-=2){
					for(j=w2-1;j>=0;j--) OP[WIDTH*i+j]=P[WIDTH*i+j]=c;
				}
			}else{
				for(i=(HEIGHT-H-FirstLine)*2;i>=0;i-=2){
					for(j=w2-1;j>=0;j--) P[WIDTH*i+j]=c;
				}
			}
		}else{
			for(i=HEIGHT-H-FirstLine;i>=0;i--){
				for(j=w2-1;j>=0;j--) P[WIDTH*i+j]=c;
			}
		}
	}
	/* Return pointer to the scanline in XBuf */
	return(P+bsiz+HAdjust);
}

/** Sprites() ************************************************/
/** This function is called from RefreshLine#() to refresh  **/
/** sprites in SCREENs 1-3.                                 **/
/*************************************************************/
void Sprites(register byte Y,register byte *Line)
{
  register byte *P,C;
  register byte H,*PT,*AT;
  register unsigned int M;
  register int L,K;

  memset((char*)Line,0,304);
  if(SpritesOFF) return;

  /* Assign initial values before counting */
  H=Sprites16x16? 16:8;
  C=0;M=0;L=0;
  AT=SprTab-4;
  Y+=VScroll;

  /* Count displayed sprites */
  do
  {
    M<<=1;AT+=4;L++;    /* Iterating through SprTab      */
    K=AT[0];            /* K = sprite Y coordinate       */
    if(K==208) break;   /* Iteration terminates if Y=208 */
    if(K>256-H) K-=256; /* Y coordinate may be negative  */

    /* Mark all valid sprites with 1s, break at MAXSPRITE1 sprites */
    if((Y>K)&&(Y<=K+H)) { M|=1;C++;if(C==MAXSPRITE1) break; }
  }
  while(L<32);

  /* Draw all marked sprites */
  for(;M;M>>=1,AT-=4)
    if(M&1)
    {
      C=AT[3];                  /* C = sprite attributes */
      L=C&0x80? AT[1]-32:AT[1]; /* Sprite may be shifted left by 32 */
      C&=0x0F;                  /* C = sprite color */

      if((L<256)&&(L>-H)&&C)
      {
        K=AT[0];                /* K = sprite Y coordinate */
        if(K>256-H) K-=256;     /* Y coordinate may be negative */

        P=Line+32+L;
        PT=SprGen+((int)(H>8? AT[2]&0xFC:AT[2])<<3)+Y-K-1;
		/* C=XPal[C]; */

        /* Mask 1: clip left sprite boundary */
        K=L>=0? 0x0FFFF:(0x10000>>-L)-1;
        /* Mask 2: clip right sprite boundary */
        if(L>256-H) K^=((0x00200>>(H-8))<<(L-257+H))-1;
        /* Get and clip the sprite data */
        K&=((int)PT[0]<<8)|(H>8? PT[16]:0x00);

        /* Draw left 8 pixels of the sprite */
        if(K&0xFF00)
        {
          if(K&0x8000) P[0]=C;if(K&0x4000) P[1]=C;
          if(K&0x2000) P[2]=C;if(K&0x1000) P[3]=C;
          if(K&0x0800) P[4]=C;if(K&0x0400) P[5]=C;
          if(K&0x0200) P[6]=C;if(K&0x0100) P[7]=C;
        }

        /* Draw right 8 pixels of the sprite */
        if(K&0x00FF)
        {
          if(K&0x0080) P[8]=C; if(K&0x0040) P[9]=C;
          if(K&0x0020) P[10]=C;if(K&0x0010) P[11]=C;
          if(K&0x0008) P[12]=C;if(K&0x0004) P[13]=C;
          if(K&0x0002) P[14]=C;if(K&0x0001) P[15]=C;
        }
      }
    }
}

/** ColorSprites() *******************************************/
/** This function is called from RefreshLine#() to refresh  **/
/** color sprites in SCREENs 4-8. The result is returned in **/
/** ZBuf, whose size must be 304 bytes (32+256+16).         **/
/*************************************************************/
void ColorSprites(register byte Y,byte *ZBuf)
{
  register byte C,H,J,OrThem;
  register byte *P,*PT,*AT;
  register int L,K;
  register unsigned int M;

  /* Clear ZBuffer and exit if sprites are off */
  memset(ZBuf+32,0,256);
  if(SpritesOFF) return;

  /* Assign initial values before counting */
  H=Sprites16x16? 16:8;
  C=0;M=0;L=0;
  AT=SprTab-4;
  OrThem=0x00;

  /* Count displayed sprites */
  do
  {
    M<<=1;AT+=4;L++;          /* Iterating through SprTab      */
    K=AT[0];                  /* Read Y from SprTab            */
    if(K==216) break;         /* Iteration terminates if Y=216 */
    K=(byte)(K-VScroll);      /* Sprite's actual Y coordinate  */
    if(K>256-H) K-=256;       /* Y coordinate may be negative  */

    /* Mark all valid sprites with 1s, break at MAXSPRITE2 sprites */
    if((Y>K)&&(Y<=K+H)) { M|=1;C++;if(C==MAXSPRITE2) break; }
  }
  while(L<32);

  /* Draw all marked sprites */
  for(;M;M>>=1,AT-=4)
    if(M&1)
    {
      K=(byte)(AT[0]-VScroll); /* K = sprite Y coordinate */
      if(K>256-H) K-=256;      /* Y coordinate may be negative */

      J=Y-K-1;
      C=SprTab[-0x0200+((AT-SprTab)<<2)+J];
      OrThem|=C&0x40;

      if(C&0x0F)
      {
        PT=SprGen+((int)(H>8? AT[2]&0xFC:AT[2])<<3)+J;
        P=ZBuf+AT[1]+(C&0x80? 0:32);
        C&=0x0F;
        J=PT[0];

        if(OrThem&0x20)
        {
          if(J&0x80) P[0]|=C;if(J&0x40) P[1]|=C;
          if(J&0x20) P[2]|=C;if(J&0x10) P[3]|=C;
          if(J&0x08) P[4]|=C;if(J&0x04) P[5]|=C;
          if(J&0x02) P[6]|=C;if(J&0x01) P[7]|=C;
          if(H>8)
          {
            J=PT[16];
            if(J&0x80) P[8]|=C; if(J&0x40) P[9]|=C;
            if(J&0x20) P[10]|=C;if(J&0x10) P[11]|=C;
            if(J&0x08) P[12]|=C;if(J&0x04) P[13]|=C;
            if(J&0x02) P[14]|=C;if(J&0x01) P[15]|=C;
          }
        }
        else
        {
          if(J&0x80) P[0]=C;if(J&0x40) P[1]=C;
          if(J&0x20) P[2]=C;if(J&0x10) P[3]=C;
          if(J&0x08) P[4]=C;if(J&0x04) P[5]=C;
          if(J&0x02) P[6]=C;if(J&0x01) P[7]=C;
          if(H>8)
          {
            J=PT[16];
            if(J&0x80) P[8]=C; if(J&0x40) P[9]=C;
            if(J&0x20) P[10]=C;if(J&0x10) P[11]=C;
            if(J&0x08) P[12]=C;if(J&0x04) P[13]=C;
            if(J&0x02) P[14]=C;if(J&0x01) P[15]=C;
          }
        }
      }

      /* Update overlapping flag */
      OrThem>>=1;
    }
}

/** RefreshLineF() *******************************************/
/** Dummy refresh function called for non-existing screens. **/
/*************************************************************/
void RefreshLineF(register byte Y)
{
  register pixel *P;

  if(Verbose>1)
    printf
    (
      "ScrMODE %d: ChrTab=%X ChrGen=%X ColTab=%X SprTab=%X SprGen=%X\n",
      ScrMode,ChrTab-VRAM,ChrGen-VRAM,ColTab-VRAM,SprTab-VRAM,SprGen-VRAM
    );

  P=RefreshBorder(Y,XPal[BGColor],256,272);
  if(P) ClearLine(P,XPal[BGColor],256);
}

/** RefreshLine0() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN0.                 **/
/*************************************************************/
void RefreshLine0(register byte Y)
{
  register pixel *P,FC,BC;
  register byte X,*T,*G;
  int i;
  
  BC=XPal[BGColor];
  P=RefreshBorder(Y,BC,256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BC,256);
  else
  {
	for(i=0;i<8+HAdjust;i++){
		*P++=BC;
	}

    G=(UseFont&&FontBuf? FontBuf:ChrGen)+((Y+VScroll)&0x07);
    T=ChrTab+40*(Y>>3);
    FC=XPal[FGColor];

    for(X=0;X<40;X++,T++,P+=6)
    {
      Y=G[(int)*T<<3];
      P[0]=Y&0x80? FC:BC;
	  P[1]=Y&0x40? FC:BC;
      P[2]=Y&0x20? FC:BC;
	  P[3]=Y&0x10? FC:BC;
      P[4]=Y&0x08? FC:BC;
	  P[5]=Y&0x04? FC:BC;
    }
	for(i=0;i<8-HAdjust;i++){
		*P++=BC;
	}
  }
}

/** RefreshLine1() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN1, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine1(register byte Y)
{
  register pixel *P,FC,BC;
  register byte K,X,XE,C,*T,*G,*R;
  byte ZBuf[304];
  int cnt;

  P=RefreshBorder(Y,XPal[BGColor],256,272);
  if(!P) return;
//  LOG( "RefreshLine1 1\n");

  if(!ScreenON) ClearLine(P,XPal[BGColor],256);
  else
  {
    Sprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    G=(UseFont&&FontBuf? FontBuf:ChrGen)+(Y&0x07);
    T=ChrTab+((int)(Y&0xF8)<<2);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;
		K=ColTab[*T>>3];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=G[(int)*T<<3];

		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x80)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x40)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x20)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x10)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x08)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x04)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x02)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x01)? FC:BC;P++; } R++;
			T++;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}
	}
	for(X=0;X<XE;X++,T++,P+=8,R+=8)
	{
	  K=ColTab[*T>>3];
	  FC=XPal[K>>4];
	  BC=XPal[K&0x0F];
	  K=G[(int)*T<<3];

	  C=R[0];P[0]=C? XPal[C]:(K&0x80)? FC:BC;
	  C=R[1];P[1]=C? XPal[C]:(K&0x40)? FC:BC;
	  C=R[2];P[2]=C? XPal[C]:(K&0x20)? FC:BC;
	  C=R[3];P[3]=C? XPal[C]:(K&0x10)? FC:BC;
	  C=R[4];P[4]=C? XPal[C]:(K&0x08)? FC:BC;
	  C=R[5];P[5]=C? XPal[C]:(K&0x04)? FC:BC;
	  C=R[6];P[6]=C? XPal[C]:(K&0x02)? FC:BC;
	  C=R[7];P[7]=C? XPal[C]:(K&0x01)? FC:BC;
	}
	if(HAdjust!=0){
		K=ColTab[*T>>3];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=G[(int)*T<<3];

		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x80)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x40)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x20)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x10)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x08)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x04)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x02)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x01)? FC:BC; cnt--;R++;P++; }
		}
	}
  }
}

/** RefreshLine2() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN2, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine2(register byte Y)
{
  register pixel *P,FC,BC;
  register byte K,X,XE,*T,*R,C;
  register int I,J;
  byte ZBuf[304];
  int cnt;

  P=RefreshBorder(Y,XPal[BGColor],256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],256);
  else
  {
    Sprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    T=ChrTab+((int)(Y&0xF8)<<2);
    I=((int)(Y&0xC0)<<5)+(Y&0x07);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;
		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];
		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x80)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x40)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x20)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x10)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x08)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x04)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x02)? FC:BC;P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x01)? FC:BC;P++; } R++;
			T++;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}
	}
	for(X=0;X<XE;X++,T++,P+=8,R+=8)
	{
		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];
		C=R[0];P[0]=C? XPal[C]:(K&0x80)? FC:BC;
		C=R[1];P[1]=C? XPal[C]:(K&0x40)? FC:BC;
		C=R[2];P[2]=C? XPal[C]:(K&0x20)? FC:BC;
		C=R[3];P[3]=C? XPal[C]:(K&0x10)? FC:BC;
		C=R[4];P[4]=C? XPal[C]:(K&0x08)? FC:BC;
		C=R[5];P[5]=C? XPal[C]:(K&0x04)? FC:BC;
		C=R[6];P[6]=C? XPal[C]:(K&0x02)? FC:BC;
		C=R[7];P[7]=C? XPal[C]:(K&0x01)? FC:BC;
	}
	if(HAdjust!=0){
		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];

		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x80)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x40)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x20)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x10)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x08)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x04)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x02)? FC:BC; cnt--;R++;P++; }
			if(cnt){ C=*R;*P=C?XPal[C]:(K&0x01)? FC:BC; cnt--;R++;P++; }
		}
	}
  }
}

/** RefreshLine3() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN3, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine3(register byte Y)
{
  register pixel *P;
  register byte X,XE,K,*T,*G,C,*R;
  byte ZBuf[304];
  int cnt;

  P=RefreshBorder(Y,XPal[BGColor],256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],256);
  else
  {
    Sprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    T=ChrTab+((int)(Y&0xF8)<<2);
    G=ChrGen+((Y&0x1C)>>2);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;
		K=G[(int)*T<<3];
		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C?C:(K>>4  )]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C?C:(K>>4  )]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C?C:(K>>4  )]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C?C:(K>>4  )]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C?C:(K&0x0F)]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C?C:(K&0x0F)]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C?C:(K&0x0F)]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C?C:(K&0x0F)]; P++; } R++;
			T++;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}
	}
	for(X=0;X<XE;X++,T++,P+=8,R+=8)
	{
	  K=G[(int)*T<<3];
	  C=R[0];P[0]=XPal[C?C:(K>>4  )];
	  C=R[1];P[1]=XPal[C?C:(K>>4  )];
	  C=R[2];P[2]=XPal[C?C:(K>>4  )];
	  C=R[3];P[3]=XPal[C?C:(K>>4  )];
	  C=R[4];P[4]=XPal[C?C:(K&0x0F)];
	  C=R[5];P[5]=XPal[C?C:(K&0x0F)];
	  C=R[6];P[6]=XPal[C?C:(K&0x0F)];
	  C=R[7];P[7]=XPal[C?C:(K&0x0F)];
	}
	if(HAdjust!=0){
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}else{
			cnt = 8 - HAdjust;
			K=G[(int)*T<<3];
			if(cnt){ C=*R;*P=XPal[C?C:(K>>4  )]; cnt--;P++; R++; }
			if(cnt){ C=*R;*P=XPal[C?C:(K>>4  )]; cnt--;P++; R++; }
			if(cnt){ C=*R;*P=XPal[C?C:(K>>4  )]; cnt--;P++; R++; }
			if(cnt){ C=*R;*P=XPal[C?C:(K>>4  )]; cnt--;P++; R++; }
			if(cnt){ C=*R;*P=XPal[C?C:(K&0x0F)]; cnt--;P++; R++; }
			if(cnt){ C=*R;*P=XPal[C?C:(K&0x0F)]; cnt--;P++; R++; }
			if(cnt){ C=*R;*P=XPal[C?C:(K&0x0F)]; cnt--;P++; R++; }
			if(cnt){ C=*R;*P=XPal[C?C:(K&0x0F)]; cnt--;P++; R++; }
		}
	}
  }
}

/** RefreshLine4() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN4, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine4(register byte Y)
{
  register pixel *P,FC,BC;
  register byte K,X,XE,C,*T,*R;
  register int I,J,cnt;
  byte ZBuf[304];

  P=RefreshBorder(Y,XPal[BGColor],256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],256);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    T=ChrTab+((int)(Y&0xF8)<<2);
    I=((int)(Y&0xC0)<<5)+(Y&0x07);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];

		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x80)? FC:BC; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x40)? FC:BC; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x20)? FC:BC; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x10)? FC:BC; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x08)? FC:BC; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x04)? FC:BC; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x02)? FC:BC; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C?XPal[C]:(K&0x01)? FC:BC; P++; } R++;
			T++;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}
	}
    for(X=0;X<XE;X++,R+=8,P+=8,T++)
    {
      J=(int)*T<<3;
      K=ColTab[(I+J)&ColTabM];
      FC=XPal[K>>4];
      BC=XPal[K&0x0F];
      K=ChrGen[(I+J)&ChrGenM];

      C=R[0];P[0]=C? XPal[C]:(K&0x80)? FC:BC;
      C=R[1];P[1]=C? XPal[C]:(K&0x40)? FC:BC;
      C=R[2];P[2]=C? XPal[C]:(K&0x20)? FC:BC;
      C=R[3];P[3]=C? XPal[C]:(K&0x10)? FC:BC;
      C=R[4];P[4]=C? XPal[C]:(K&0x08)? FC:BC;
      C=R[5];P[5]=C? XPal[C]:(K&0x04)? FC:BC;
      C=R[6];P[6]=C? XPal[C]:(K&0x02)? FC:BC;
      C=R[7];P[7]=C? XPal[C]:(K&0x01)? FC:BC;
    }

	if(HAdjust!=0){
		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];

		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				*P = XPal[BGColor]; P++;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ C=*R;*P=C? XPal[C]:(K&0x80)? FC:BC; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=C? XPal[C]:(K&0x40)? FC:BC; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=C? XPal[C]:(K&0x20)? FC:BC; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=C? XPal[C]:(K&0x10)? FC:BC; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=C? XPal[C]:(K&0x08)? FC:BC; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=C? XPal[C]:(K&0x04)? FC:BC; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=C? XPal[C]:(K&0x02)? FC:BC; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=C? XPal[C]:(K&0x01)? FC:BC; cnt--; R++;P++; }
		}
	}

  }
}

/** RefreshLine5() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN5, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine5(register byte Y)
{
  register pixel *P;
  register byte I,X,XE,*T,*R;
  register int cnt;
  byte ZBuf[304];

  P=RefreshBorder(Y,XPal[BGColor],256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],256);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<7)&ChrTabM&0x7FFF);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ I=*R;*P=XPal[I? I:T[0]>>4];   P++; } R++;
			if(cnt){cnt--;}else{ I=*R;*P=XPal[I? I:T[0]&0x0F]; P++; } R++;
			if(cnt){cnt--;}else{ I=*R;*P=XPal[I? I:T[1]>>4];   P++; } R++;
			if(cnt){cnt--;}else{ I=*R;*P=XPal[I? I:T[1]&0x0F]; P++; } R++;
			if(cnt){cnt--;}else{ I=*R;*P=XPal[I? I:T[2]>>4];   P++; } R++;
			if(cnt){cnt--;}else{ I=*R;*P=XPal[I? I:T[2]&0x0F]; P++; } R++;
			if(cnt){cnt--;}else{ I=*R;*P=XPal[I? I:T[3]>>4];   P++; } R++;
			if(cnt){cnt--;}else{ I=*R;*P=XPal[I? I:T[3]&0x0F]; P++; } R++;
			T+=4;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}
	}

    for(X=0;X<XE;X++,R+=8,P+=8,T+=4)
    {
      I=R[ 0];P[ 0]=XPal[I? I:T[0]>>4];
      I=R[ 1];P[ 1]=XPal[I? I:T[0]&0x0F];
      I=R[ 2];P[ 2]=XPal[I? I:T[1]>>4];
      I=R[ 3];P[ 3]=XPal[I? I:T[1]&0x0F];
      I=R[ 4];P[ 4]=XPal[I? I:T[2]>>4];
      I=R[ 5];P[ 5]=XPal[I? I:T[2]&0x0F];
      I=R[ 6];P[ 6]=XPal[I? I:T[3]>>4];
      I=R[ 7];P[ 7]=XPal[I? I:T[3]&0x0F];
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ I=*R;*P=XPal[I? I:T[0]>>4];   cnt--; R++;P++; }
			if(cnt){ I=*R;*P=XPal[I? I:T[0]&0x0F]; cnt--; R++;P++; }
			if(cnt){ I=*R;*P=XPal[I? I:T[1]>>4];   cnt--; R++;P++; }
			if(cnt){ I=*R;*P=XPal[I? I:T[1]&0x0F]; cnt--; R++;P++; }
			if(cnt){ I=*R;*P=XPal[I? I:T[2]>>4];   cnt--; R++;P++; }
			if(cnt){ I=*R;*P=XPal[I? I:T[2]&0x0F]; cnt--; R++;P++; }
			if(cnt){ I=*R;*P=XPal[I? I:T[3]>>4];   cnt--; R++;P++; }
			if(cnt){ I=*R;*P=XPal[I? I:T[3]&0x0F]; cnt--; R++;P++; }
		}
	}
  }
}

/** RefreshLine6() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN6, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine6(register byte Y)
{
  register pixel *P;
  register byte X,XE,*T,*R,C;
  register int cnt;
  byte ZBuf[304];

  P=RefreshBorder(Y,XPal[BGColor&0x03],256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor&0x03],256);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<7)&ChrTabM&0x7FFF);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[0]>>6];        P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[0]>>2)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[1]>>6];        P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[1]>>2)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[2]>>6];        P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[2]>>2)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[3]>>6];        P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[3]>>2)&0x03]; P++; } R++;
			T+=4;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor&0x03];P++;
			}
		}
	}
    for(X=0;X<XE;X++)
    {
      C=R[0];P[0]=XPal[C? C:T[0]>>6];
      C=R[1];P[1]=XPal[C? C:(T[0]>>2)&0x03];
      C=R[2];P[2]=XPal[C? C:T[1]>>6];
      C=R[3];P[3]=XPal[C? C:(T[1]>>2)&0x03];
      C=R[4];P[4]=XPal[C? C:T[2]>>6];
      C=R[5];P[5]=XPal[C? C:(T[2]>>2)&0x03];
      C=R[6];P[6]=XPal[C? C:T[3]>>6];
      C=R[7];P[7]=XPal[C? C:(T[3]>>2)&0x03];
      R+=8;P+=8;T+=4;
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				*P = XPal[BGColor&0x03];P++;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ C=*R;*P=XPal[C? C:T[0]>>6];        cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[0]>>2)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[1]>>6];        cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[1]>>2)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[2]>>6];        cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[2]>>2)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[3]>>6];        cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[3]>>2)&0x03]; cnt--; R++;P++; }
		}
	}
  }
}

/** RefreshLine6W() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN6, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine6W(register byte Y)
{
  register pixel *P;
  register byte X,XE,*T,*R,C;
  register int cnt;
  byte ZBuf[304];

  P=RefreshBorder(Y,XPal[BGColor&0x03],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor&0x03],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<7)&ChrTabM&0x7FFF);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		if( HAdjust<0 ){
			cnt = -HAdjust*2;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C: T[0]>>6];       P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[0]>>4)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[0]>>2)&0x03]; P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[0]   )&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C: T[1]>>6];       P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[1]>>4)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[1]>>2)&0x03]; P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[1]   )&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C: T[2]>>6];       P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[2]>>4)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[2]>>2)&0x03]; P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[2]   )&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C: T[3]>>6      ]; P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[3]>>4)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[3]>>2)&0x03]; P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[3]   )&0x03]; P++; } R++;
			T+=4;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor&0x03];P++;
			}
		}
	}
    for(X=0;X<XE;X++)
    {
      C=R[0];P[ 0]=XPal[C? C:(T[0]>>6)];
      C=R[0];P[ 1]=XPal[C? C:(T[0]>>4)&0x03];
      C=R[1];P[ 2]=XPal[C? C:(T[0]>>2)&0x03];
      C=R[1];P[ 3]=XPal[C? C: T[0]    &0x03];
      C=R[2];P[ 4]=XPal[C? C:(T[1]>>6)];
      C=R[2];P[ 5]=XPal[C? C:(T[1]>>4)&0x03];
      C=R[3];P[ 6]=XPal[C? C:(T[1]>>2)&0x03];
      C=R[3];P[ 7]=XPal[C? C: T[1]    &0x03];
      C=R[4];P[ 8]=XPal[C? C:(T[2]>>6)];
      C=R[4];P[ 9]=XPal[C? C:(T[2]>>4)&0x03];
      C=R[5];P[10]=XPal[C? C:(T[2]>>2)&0x03];
      C=R[5];P[11]=XPal[C? C: T[2]    &0x03];
      C=R[6];P[12]=XPal[C? C:(T[3]>>6)];
      C=R[6];P[13]=XPal[C? C:(T[3]>>4)&0x03];
      C=R[7];P[14]=XPal[C? C:(T[3]>>2)&0x03];
      C=R[7];P[15]=XPal[C? C: T[3]    &0x03];
      R+=8;P+=16;T+=4;
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust*2;X++ ){
				*P = XPal[BGColor&0x03];P++;
			}
		}else{
			cnt = 16 - HAdjust*2;
			if(cnt){ C=*R;*P=XPal[C? C: T[0]>>6      ]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[0]>>4)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[0]>>2)&0x03]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[0]   )&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C: T[1]>>6      ]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[1]>>4)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[1]>>2)&0x03]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[1]   )&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C: T[2]>>6      ]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[2]>>4)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[2]>>2)&0x03]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[2]   )&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C: T[3]>>6      ]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[3]>>4)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[3]>>2)&0x03]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[3]   )&0x03]; cnt--; R++;P++; }
		}
	}
  }
}
  
/** RefreshLine7() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN7, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine7(register byte Y)
{
  register pixel *P;
  register byte C,X,XE,*T,*R;
  register int cnt;
  byte ZBuf[304];

  P=RefreshBorder(Y,XPal[BGColor],256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],256);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[0]>>4]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[1]>>4]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[2]>>4]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[3]>>4]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[4]>>4]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[5]>>4]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[6]>>4]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[7]>>4]; P++; } R++;
			T+= 8;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}
	}
    for(X=0;X<XE;X++)
    {
      C=R[0];P[0]=XPal[C? C:T[0]>>4];
      C=R[1];P[1]=XPal[C? C:T[1]>>4];
      C=R[2];P[2]=XPal[C? C:T[2]>>4];
      C=R[3];P[3]=XPal[C? C:T[3]>>4];
      C=R[4];P[4]=XPal[C? C:T[4]>>4];
      C=R[5];P[5]=XPal[C? C:T[5]>>4];
      C=R[6];P[6]=XPal[C? C:T[6]>>4];
      C=R[7];P[7]=XPal[C? C:T[7]>>4];
      R+=8;P+=8;T+=8;
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ C=*R;*P=XPal[C? C:T[0]>>4]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[1]>>4]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[2]>>4]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[3]>>4]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[4]>>4]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[5]>>4]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[6]>>4]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[7]>>4]; cnt--;P++;R++; }
		}
	}
  }
}

/** RefreshLine7W() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN7, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine7W(register byte Y)
{
  register pixel *P;
  register byte C,X,XE,*T,*R;
  register int cnt;
  byte ZBuf[304];

  P=RefreshBorder(Y,XPal[BGColor],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		if( HAdjust<0 ){
			cnt = -HAdjust*2;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[0]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[0]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[1]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[1]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[2]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[2]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[3]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[3]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[4]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[4]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[5]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[5]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[6]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[6]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[7]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[7]&0xF]; P++; } R++;
			T+= 8;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}
	}
    for(X=0;X<XE;X++)
    {
      C=R[0];P[ 0]=XPal[C? C:T[0]>>4];
      C=R[0];P[ 1]=XPal[C? C:T[0]   &0xF];
      C=R[1];P[ 2]=XPal[C? C:T[1]>>4];
      C=R[1];P[ 3]=XPal[C? C:T[1]   &0xF];
      C=R[2];P[ 4]=XPal[C? C:T[2]>>4];
      C=R[2];P[ 5]=XPal[C? C:T[2]   &0xF];
      C=R[3];P[ 6]=XPal[C? C:T[3]>>4];
      C=R[3];P[ 7]=XPal[C? C:T[3]   &0xF];
      C=R[4];P[ 8]=XPal[C? C:T[4]>>4];
      C=R[4];P[ 9]=XPal[C? C:T[4]   &0xF];
      C=R[5];P[10]=XPal[C? C:T[5]>>4];
      C=R[5];P[11]=XPal[C? C:T[5]   &0xF];
      C=R[6];P[12]=XPal[C? C:T[6]>>4];
      C=R[6];P[13]=XPal[C? C:T[6]   &0xF];
      C=R[7];P[14]=XPal[C? C:T[7]>>4];
      C=R[7];P[15]=XPal[C? C:T[7]   &0xF];
      R+=8;P+=16;T+=8;
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust*2;X++ ){
				*P = XPal[BGColor];P++;
			}
		}else{
			cnt = 16 - HAdjust*2;
			if(cnt){ C=*R;*P=XPal[C? C:T[0]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[0]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[1]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[1]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[2]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[2]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[3]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[3]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[4]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[4]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[5]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[5]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[6]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[6]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[7]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[7]&0xF]; cnt--;P++;R++; }
		}
	}
  }
}

/** RefreshLine8() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN8, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine8(register byte Y)
{
  static byte SprToScr[16] =
  {
    0x00,0x02,0x10,0x12,0x80,0x82,0x90,0x92,
    0x49,0x4B,0x59,0x5B,0xC9,0xCB,0xD9,0xDB
  };
  register pixel *P;
  register byte C,X,XE,*T,*R;
  register int cnt;
  byte ZBuf[304];

  P=RefreshBorder(Y,BPal[VDP[7]],256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BPal[VDP[7]],256);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;*P=BPal[C? SprToScr[C]:T[0]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=BPal[C? SprToScr[C]:T[1]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=BPal[C? SprToScr[C]:T[2]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=BPal[C? SprToScr[C]:T[3]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=BPal[C? SprToScr[C]:T[4]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=BPal[C? SprToScr[C]:T[5]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=BPal[C? SprToScr[C]:T[6]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=BPal[C? SprToScr[C]:T[7]]; P++; } R++;
			T+= 8;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = BPal[VDP[7]];P++;
			}
		}
	}
    for(X=0;X<XE;X++,T+=8,R+=8,P+=8)
    {
      C=R[0];P[0]=BPal[C? SprToScr[C]:T[0]];
      C=R[1];P[1]=BPal[C? SprToScr[C]:T[1]];
      C=R[2];P[2]=BPal[C? SprToScr[C]:T[2]];
      C=R[3];P[3]=BPal[C? SprToScr[C]:T[3]];
      C=R[4];P[4]=BPal[C? SprToScr[C]:T[4]];
      C=R[5];P[5]=BPal[C? SprToScr[C]:T[5]];
      C=R[6];P[6]=BPal[C? SprToScr[C]:T[6]];
      C=R[7];P[7]=BPal[C? SprToScr[C]:T[7]];
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				*P = BPal[VDP[7]];P++;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ C=*R;*P=BPal[C? SprToScr[C]:T[0]]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=BPal[C? SprToScr[C]:T[1]]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=BPal[C? SprToScr[C]:T[2]]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=BPal[C? SprToScr[C]:T[3]]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=BPal[C? SprToScr[C]:T[4]]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=BPal[C? SprToScr[C]:T[5]]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=BPal[C? SprToScr[C]:T[6]]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=BPal[C? SprToScr[C]:T[7]]; cnt--;P++;R++; }
		}
	}
  }
}

/** RefreshLine10() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN10/11, including   **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine10(register byte Y)
{
  register pixel *P;
  register byte C,X,XE,*T,*R;
  register int J,K,cnt;
  byte ZBuf[304];

  P=RefreshBorder(Y,BPal[VDP[7]],256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BPal[VDP[7]],256);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

	if(HAdjust==0){
		XE=63;
		/* Draw first 4 pixels */
		C=R[0];P[0]=C? XPal[C]:BPal[VDP[7]];
		C=R[1];P[1]=C? XPal[C]:BPal[VDP[7]];
		C=R[2];P[2]=C? XPal[C]:BPal[VDP[7]];
		C=R[3];P[3]=C? XPal[C]:BPal[VDP[7]];
		R+=4;P+=4;
	}else {
		XE=62;

		if( HAdjust<0 ){

			cnt = -HAdjust;
		    /* Draw first 4 pixels */
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:BPal[VDP[7]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:BPal[VDP[7]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:BPal[VDP[7]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:BPal[VDP[7]]; P++; } R++;

			K=(T[0]&0x07)|((T[1]&0x07)<<3);
			if(K&0x20) K-=64;
			J=(T[2]&0x07)|((T[3]&0x07)<<3);
			if(J&0x20) J-=64;

			if(cnt){cnt--;}else{ C=*R;Y=T[0]>>3;*P=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); P++; } R++;
			if(cnt){cnt--;}else{ C=*R;Y=T[1]>>3;*P=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); P++; } R++;
			if(cnt){cnt--;}else{ C=*R;Y=T[2]>>3;*P=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); P++; } R++;
			if(cnt){cnt--;}else{ C=*R;Y=T[3]>>3;*P=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); P++; } R++;
			T+=4;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = BPal[VDP[7]];P++;
			}
			C=R[0];P[0]=C? XPal[C]:BPal[VDP[7]];
			C=R[1];P[1]=C? XPal[C]:BPal[VDP[7]];
			C=R[2];P[2]=C? XPal[C]:BPal[VDP[7]];
			C=R[3];P[3]=C? XPal[C]:BPal[VDP[7]];
			R+=4;P+=4;
		}
	}
    for(X=0;X<XE;X++,T+=4,R+=4,P+=4)
    {
      K=(T[0]&0x07)|((T[1]&0x07)<<3);
      if(K&0x20) K-=64;
      J=(T[2]&0x07)|((T[3]&0x07)<<3);
      if(J&0x20) J-=64;

      C=R[0];Y=T[0]>>3;P[0]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
      C=R[1];Y=T[1]>>3;P[1]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
      C=R[2];Y=T[2]>>3;P[2]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
      C=R[3];Y=T[3]>>3;P[3]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<HAdjust;X++ ){
				*P = BPal[VDP[7]];P++;
			}
		}else{
			cnt = 8 - HAdjust;
			for(X=0;X<2;X++){
				K=(T[0]&0x07)|((T[1]&0x07)<<3);
				if(K&0x20) K-=64;
				J=(T[2]&0x07)|((T[3]&0x07)<<3);
				if(J&0x20) J-=64;
				if(cnt){ C=R[0];Y=T[0]>>3;P[0]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); cnt--; } else break;
				if(cnt){ C=R[1];Y=T[1]>>3;P[1]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); cnt--; } else break;
				if(cnt){ C=R[2];Y=T[2]>>3;P[2]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); cnt--; } else break;
				if(cnt){ C=R[3];Y=T[3]>>3;P[3]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); cnt--; } else break;
				T+=4;R+=4;P+=4;
			}
		}
	}
  }
}

/** RefreshLine12() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN12, including      **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine12(register byte Y)
{
  register pixel *P;
  register byte C,X,XE,*T,*R;
  register int J,K,cnt;
  byte ZBuf[304];

  P=RefreshBorder(Y,BPal[VDP[7]],256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BPal[VDP[7]],256);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

    if(HScroll512&&(HScroll>255)) T=(byte *)((int)T^0x10000);
    T+=HScroll&0xFC;

	if(HAdjust==0){
		XE=63;
		/* Draw first 4 pixels */
		C=R[0];P[0]=C? XPal[C]:BPal[VDP[7]];
		C=R[1];P[1]=C? XPal[C]:BPal[VDP[7]];
		C=R[2];P[2]=C? XPal[C]:BPal[VDP[7]];
		C=R[3];P[3]=C? XPal[C]:BPal[VDP[7]];
		R+=4;P+=4;
	}else {
		XE=62;

		if( HAdjust<0 ){
			cnt = -HAdjust;
			/* Draw first 4 pixels */
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:BPal[VDP[7]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:BPal[VDP[7]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:BPal[VDP[7]]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:BPal[VDP[7]]; P++; } R++;

			K=(T[0]&0x07)|((T[1]&0x07)<<3);
			if(K&0x20) K-=64;
			J=(T[2]&0x07)|((T[3]&0x07)<<3);
			if(J&0x20) J-=64;

			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:YJKColor(T[0]>>3,J,K); P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:YJKColor(T[1]>>3,J,K); P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:YJKColor(T[2]>>3,J,K); P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=C? XPal[C]:YJKColor(T[3]>>3,J,K); P++; } R++;
			T++;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = BPal[VDP[7]];P++;
			}
			/* Draw first 4 pixels */
			C=R[0];P[0]=C? XPal[C]:BPal[VDP[7]];
			C=R[1];P[1]=C? XPal[C]:BPal[VDP[7]];
			C=R[2];P[2]=C? XPal[C]:BPal[VDP[7]];
			C=R[3];P[3]=C? XPal[C]:BPal[VDP[7]];
			R+=4;P+=4;
		}
	}
    for(X=0;X<XE;X++,T+=4,R+=4,P+=4)
    {
      K=(T[0]&0x07)|((T[1]&0x07)<<3);
      if(K&0x20) K-=64;
      J=(T[2]&0x07)|((T[3]&0x07)<<3);
      if(J&0x20) J-=64;

      C=R[0];P[0]=C? XPal[C]:YJKColor(T[0]>>3,J,K);
      C=R[1];P[1]=C? XPal[C]:YJKColor(T[1]>>3,J,K);
      C=R[2];P[2]=C? XPal[C]:YJKColor(T[2]>>3,J,K);
      C=R[3];P[3]=C? XPal[C]:YJKColor(T[3]>>3,J,K);
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<HAdjust;X++ ){
				*P = BPal[VDP[7]];P++;
			}
		}else{
			cnt = 8 - HAdjust;
			for(X=0;X<2;X++){
				K=(T[0]&0x07)|((T[1]&0x07)<<3);
				if(K&0x20) K-=64;
				J=(T[2]&0x07)|((T[3]&0x07)<<3);
				if(J&0x20) J-=64;
				if(cnt){ C=R[0];P[0]=C? XPal[C]:YJKColor(T[0]>>3,J,K); cnt--; }else break;
				if(cnt){ C=R[1];P[1]=C? XPal[C]:YJKColor(T[1]>>3,J,K); cnt--; }else break;
				if(cnt){ C=R[2];P[2]=C? XPal[C]:YJKColor(T[2]>>3,J,K); cnt--; }else break;
				if(cnt){ C=R[3];P[3]=C? XPal[C]:YJKColor(T[3]>>3,J,K); cnt--; }else break;
				T+=4,R+=4,P+=4;
			}
		}
	}
  }
}


/** RefreshLineTx80() ****************************************/
/** Refresh line Y (0..191/211) of TEXT80.                  **/
/*************************************************************/
void RefreshLineTx80(register byte Y)
{
  register pixel *P,FC,BC;
  register byte X,M,*T,*C,*G;
  register int i;

  BC=XPal[BGColor];
  P=RefreshBorder(Y,BC,256,272);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BC,256);
  else
  {
	for(i=0;i<8+HAdjust;i++){
		*P++=BC;
	}
    G=(UseFont&&FontBuf? FontBuf:ChrGen)+((Y+VScroll)&0x07);
    T=ChrTab+((80*(Y>>3))&ChrTabM);
    C=ColTab+((10*(Y>>3))&ColTabM);
    P+=9;

    for(X=0,M=0x00;X<80;X++,T++,P+=3)
    {
      if(!(X&0x07)) M=*C++;
      if(M&0x80) { FC=XPal[XFGColor];BC=XPal[XBGColor]; }
      else       { FC=XPal[FGColor];BC=XPal[BGColor]; }
      M<<=1;
      Y=*(G+((int)*T<<3));
      P[0]=Y&0xC0? FC:BC;
      P[1]=Y&0x30? FC:BC;
      P[2]=Y&0x0C? FC:BC;
    }
	for(i=0;i<8-HAdjust;i++){
		*P++=XPal[BGColor];
	}
  }
}


/** RefreshLineTx80W() ****************************************/
/** Refresh line Y (0..191/211) of TEXT80.                  **/
/*************************************************************/
void RefreshLineTx80W(register byte Y)
{
  register pixel *P,FC,BC;
  register byte X,M,*T,*C,*G;
  register int i;

  BC=XPal[BGColor];
  P=RefreshBorder(Y,BC,512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BC,512);
  else
  {
	for(i=0;i<8+HAdjust;i++){
		*P++=BC;
	}
    G=(UseFont&&FontBuf? FontBuf:ChrGen)+((Y+VScroll)&0x07);
    T=ChrTab+80*(Y>>3);
    C=ColTab+10*(Y>>3);
    P+=9;

    for(X=0,M=0x00;X<80;X++,T++,P+=6)
    {
      if(!(X&0x07)) M=*C++;
      if(M&0x80) { FC=XPal[XFGColor];BC=XPal[XBGColor]; }
      else       { FC=XPal[ FGColor];BC=XPal[BGColor]; }
      M<<=1;
      Y=*(G+((int)*T<<3));
      P[0]=Y&0x80? FC:BC;
      P[1]=Y&0x40? FC:BC;
      P[2]=Y&0x20? FC:BC;
      P[3]=Y&0x10? FC:BC;
      P[4]=Y&0x08? FC:BC;
      P[5]=Y&0x04? FC:BC;
    }
	for(i=0;i<8-HAdjust;i++){
		*P++=XPal[BGColor];
	}
  }
}

/** RefreshLineF() *******************************************/
/** Dummy refresh function called for non-existing screens. **/
/*************************************************************/
void RefreshLineF_x2(register byte Y)
{
  register pixel *P;

  if(Verbose>1)
    LOG
    (
      "ScrMODE %d: ChrTab=%X ChrGen=%X ColTab=%X SprTab=%X SprGen=%X\n",
      ScrMode,ChrTab-VRAM,ChrGen-VRAM,ColTab-VRAM,SprTab-VRAM,SprGen-VRAM
    );

  P=RefreshBorder(Y,XPal[BGColor],512,528);
  if(P) ClearLine(P,XPal[BGColor],512);
}

/** RefreshLine0() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN0.                 **/
/*************************************************************/
void RefreshLine0_x2(register byte Y)
{
  register pixel *P,*OP,FC,BC;
  register byte X,*T,*G;
  int i;
  BC=XPal[BGColor];
  OP = P =RefreshBorder(Y,BC,512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BC,512);
  else
  {
	for(i=0;i<(8+HAdjust)*2;i++){
		*P++=BC;
	}

    G=(UseFont&&FontBuf? FontBuf:ChrGen)+((Y+VScroll)&0x07);
    T=ChrTab+40*(Y>>3);
    FC=XPal[FGColor];
    P+=9;

    for(X=0;X<40;X++,T++,P+=12)
    {
      Y=G[(int)*T<<3];
      P[ 0]=P[ 1]=Y&0x80? FC:BC;
	  P[ 2]=P[ 3]=Y&0x40? FC:BC;
      P[ 4]=P[ 5]=Y&0x20? FC:BC;
	  P[ 6]=P[ 7]=Y&0x10? FC:BC;
      P[ 8]=P[ 9]=Y&0x08? FC:BC;
	  P[10]=P[11]=Y&0x04? FC:BC;
    }
	for(i=0;i<(8-HAdjust)*2;i++){
		*P++=BC;
	}
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine1() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN1, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine1_x2(register byte Y)
{
  register pixel *P,*OP,FC,BC;
  register byte K,X,XE,*T,*G,*R,C;
  register int cnt;
  byte ZBuf[304];

  OP = P=RefreshBorder(Y,XPal[BGColor],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    Sprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    G=(UseFont&&FontBuf? FontBuf:ChrGen)+(Y&0x07);
    T=ChrTab+((int)(Y&0xF8)<<2);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;
		K=ColTab[*T>>3];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=G[(int)*T<<3];

		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x80)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x40)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x20)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x10)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x08)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x04)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x02)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x01)? FC:BC;P+=2; } R++;
			T++;
		}else{
			for(X=0;X<HAdjust;X++ ){
				P[0]=P[1]=XPal[BGColor];P+=2;
			}
		}
	}
    for(X=0;X<XE;X++,T++,P+=16,R+=8)
    {
      K=ColTab[*T>>3];
      FC=XPal[K>>4];
      BC=XPal[K&0x0F];
      K=G[(int)*T<<3];

	  C=R[0];P[ 0]=P[ 1]=C? XPal[C]:(K&0x80)? FC:BC;
	  C=R[1];P[ 2]=P[ 3]=C? XPal[C]:(K&0x40)? FC:BC;
	  C=R[2];P[ 4]=P[ 5]=C? XPal[C]:(K&0x20)? FC:BC;
	  C=R[3];P[ 6]=P[ 7]=C? XPal[C]:(K&0x10)? FC:BC;
	  C=R[4];P[ 8]=P[ 9]=C? XPal[C]:(K&0x08)? FC:BC;
	  C=R[5];P[10]=P[11]=C? XPal[C]:(K&0x04)? FC:BC;
	  C=R[6];P[12]=P[13]=C? XPal[C]:(K&0x02)? FC:BC;
	  C=R[7];P[14]=P[15]=C? XPal[C]:(K&0x01)? FC:BC;
    }
	if(HAdjust!=0){
		K=ColTab[*T>>3];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=G[(int)*T<<3];

		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				P[0]=P[1]=XPal[BGColor];P+=2;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x80)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x40)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x20)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x10)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x08)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x04)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x02)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x01)? FC:BC; cnt--;R++;P+=2; }
		}
	}
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine2() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN2, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine2_x2(register byte Y)
{
  register pixel *P,*OP,FC,BC;
  register byte K,X,XE,*T,*R,C;
  register int I,J;
  byte ZBuf[304];
  int cnt;

  OP=P=RefreshBorder(Y,XPal[BGColor],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    Sprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    T=ChrTab+((int)(Y&0xF8)<<2);
    I=((int)(Y&0xC0)<<5)+(Y&0x07);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;
		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];
		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x80)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x40)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x20)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x10)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x08)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x04)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x02)? FC:BC;P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x01)? FC:BC;P+=2; } R++;
			T++;
		}else{
			for(X=0;X<HAdjust;X++ ){
				P[0]=P[1]=XPal[BGColor];P+=2;
			}
		}
	}
    for(X=0;X<XE;X++,T++,P+=16,R+=8)
    {
		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];
		C=R[0];P[ 0]=P[ 1]=C? XPal[C]:(K&0x80)? FC:BC;
		C=R[1];P[ 2]=P[ 3]=C? XPal[C]:(K&0x40)? FC:BC;
		C=R[2];P[ 4]=P[ 5]=C? XPal[C]:(K&0x20)? FC:BC;
		C=R[3];P[ 6]=P[ 7]=C? XPal[C]:(K&0x10)? FC:BC;
		C=R[4];P[ 8]=P[ 9]=C? XPal[C]:(K&0x08)? FC:BC;
		C=R[5];P[10]=P[11]=C? XPal[C]:(K&0x04)? FC:BC;
		C=R[6];P[12]=P[13]=C? XPal[C]:(K&0x02)? FC:BC;
		C=R[7];P[14]=P[15]=C? XPal[C]:(K&0x01)? FC:BC;
    }
	if(HAdjust!=0){
		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];

		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				P[0]=P[1]= XPal[BGColor];P+=2;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x80)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x40)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x20)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x10)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x08)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x04)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x02)? FC:BC; cnt--;R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x01)? FC:BC; cnt--;R++;P+=2; }
		}
	}
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine3() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN3, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine3_x2(register byte Y)
{
  register pixel *P,*OP;
  register byte X,XE,K,*T,*G,*R,C;
  byte ZBuf[304];
  int cnt;

  OP=P=RefreshBorder(Y,XPal[BGColor],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    Sprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    T=ChrTab+((int)(Y&0xF8)<<2);
    G=ChrGen+((Y&0x1C)>>2);
	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;
		K=G[(int)*T<<3];
		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=XPal[C?C:(K>>4  )]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=XPal[C?C:(K>>4  )]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=XPal[C?C:(K>>4  )]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=XPal[C?C:(K>>4  )]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=XPal[C?C:(K&0x0F)]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=XPal[C?C:(K&0x0F)]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=XPal[C?C:(K&0x0F)]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=XPal[C?C:(K&0x0F)]; P+=2; } R++;
			T++;
		}else{
			for(X=0;X<HAdjust;X++ ){
				P[0]=P[1]= XPal[BGColor];P+=2;
			}
		}
	}
    for(X=0;X<XE;X++,T++,P+=16)
    {
      K=G[(int)*T<<3];
	  C=R[0];P[ 0]=P[ 1]=XPal[C?C:(K>>4  )];
	  C=R[1];P[ 2]=P[ 3]=XPal[C?C:(K>>4  )];
	  C=R[2];P[ 4]=P[ 5]=XPal[C?C:(K>>4  )];
	  C=R[3];P[ 6]=P[ 7]=XPal[C?C:(K>>4  )];
	  C=R[4];P[ 8]=P[ 9]=XPal[C?C:(K&0x0F)];
	  C=R[5];P[10]=P[11]=XPal[C?C:(K&0x0F)];
	  C=R[6];P[12]=P[13]=XPal[C?C:(K&0x0F)];
	  C=R[7];P[14]=P[15]=XPal[C?C:(K&0x0F)];
    }
	if(HAdjust!=0){
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				P[0]=P[1]= XPal[BGColor];P+=2;
			}
		}else{
			cnt = 8 - HAdjust;
			K=G[(int)*T<<3];
			if(cnt){ C=*R;P[0]=P[1]=XPal[C?C:(K>>4  )]; cnt--;P+=2; R++; }
			if(cnt){ C=*R;P[0]=P[1]=XPal[C?C:(K>>4  )]; cnt--;P+=2; R++; }
			if(cnt){ C=*R;P[0]=P[1]=XPal[C?C:(K>>4  )]; cnt--;P+=2; R++; }
			if(cnt){ C=*R;P[0]=P[1]=XPal[C?C:(K>>4  )]; cnt--;P+=2; R++; }
			if(cnt){ C=*R;P[0]=P[1]=XPal[C?C:(K&0x0F)]; cnt--;P+=2; R++; }
			if(cnt){ C=*R;P[0]=P[1]=XPal[C?C:(K&0x0F)]; cnt--;P+=2; R++; }
			if(cnt){ C=*R;P[0]=P[1]=XPal[C?C:(K&0x0F)]; cnt--;P+=2; R++; }
			if(cnt){ C=*R;P[0]=P[1]=XPal[C?C:(K&0x0F)]; cnt--;P+=2; R++; }
		}
	}
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine4() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN4, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine4_x2(register byte Y)
{
  register pixel *P,*OP,FC,BC;
  register byte K,X,C,*T,*R,XE;
  register int I,J,cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,XPal[BGColor],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    Y+=VScroll;
    T=ChrTab+((int)(Y&0xF8)<<2);
    I=((int)(Y&0xC0)<<5)+(Y&0x07);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];

		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x80)? FC:BC; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x40)? FC:BC; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x20)? FC:BC; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x10)? FC:BC; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x08)? FC:BC; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x04)? FC:BC; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x02)? FC:BC; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C?XPal[C]:(K&0x01)? FC:BC; P+=2; } R++;
			T++;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}
	}

    for(X=0;X<XE;X++,R+=8,P+=16,T++)
    {
      J=(int)*T<<3;
      K=ColTab[(I+J)&ColTabM];
      FC=XPal[K>>4];
      BC=XPal[K&0x0F];
      K=ChrGen[(I+J)&ChrGenM];

      C=R[0];P[ 0]=P[ 1]=C? XPal[C]:(K&0x80)? FC:BC;
      C=R[1];P[ 2]=P[ 3]=C? XPal[C]:(K&0x40)? FC:BC;
      C=R[2];P[ 4]=P[ 5]=C? XPal[C]:(K&0x20)? FC:BC;
      C=R[3];P[ 6]=P[ 7]=C? XPal[C]:(K&0x10)? FC:BC;
      C=R[4];P[ 8]=P[ 9]=C? XPal[C]:(K&0x08)? FC:BC;
      C=R[5];P[10]=P[11]=C? XPal[C]:(K&0x04)? FC:BC;
      C=R[6];P[12]=P[13]=C? XPal[C]:(K&0x02)? FC:BC;
      C=R[7];P[14]=P[15]=C? XPal[C]:(K&0x01)? FC:BC;
    }
	if(HAdjust!=0){
		J=(int)*T<<3;
		K=ColTab[(I+J)&ColTabM];
		FC=XPal[K>>4];
		BC=XPal[K&0x0F];
		K=ChrGen[(I+J)&ChrGenM];

		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				*P = XPal[BGColor]; P++;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ C=*R;P[0]=P[1]=C? XPal[C]:(K&0x80)? FC:BC; cnt--; R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C? XPal[C]:(K&0x40)? FC:BC; cnt--; R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C? XPal[C]:(K&0x20)? FC:BC; cnt--; R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C? XPal[C]:(K&0x10)? FC:BC; cnt--; R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C? XPal[C]:(K&0x08)? FC:BC; cnt--; R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C? XPal[C]:(K&0x04)? FC:BC; cnt--; R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C? XPal[C]:(K&0x02)? FC:BC; cnt--; R++;P+=2; }
			if(cnt){ C=*R;P[0]=P[1]=C? XPal[C]:(K&0x01)? FC:BC; cnt--; R++;P+=2; }
		}
	}
    
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine5() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN5, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine5_x2(register byte Y)
{
  register pixel *P,*OP;
  register byte I,X,XE,*T,*R;
  byte ZBuf[304];
  int cnt;

  OP=P=RefreshBorder(Y,XPal[BGColor],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<7)&ChrTabM&0x7FFF);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;
		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ I=*R;P[0]=P[1]=XPal[I? I:T[0]>>4];   P+=2; } R++;
			if(cnt){cnt--;}else{ I=*R;P[0]=P[1]=XPal[I? I:T[0]&0x0F]; P+=2; } R++;
			if(cnt){cnt--;}else{ I=*R;P[0]=P[1]=XPal[I? I:T[1]>>4];   P+=2; } R++;
			if(cnt){cnt--;}else{ I=*R;P[0]=P[1]=XPal[I? I:T[1]&0x0F]; P+=2; } R++;
			if(cnt){cnt--;}else{ I=*R;P[0]=P[1]=XPal[I? I:T[2]>>4];   P+=2; } R++;
			if(cnt){cnt--;}else{ I=*R;P[0]=P[1]=XPal[I? I:T[2]&0x0F]; P+=2; } R++;
			if(cnt){cnt--;}else{ I=*R;P[0]=P[1]=XPal[I? I:T[3]>>4];   P+=2; } R++;
			if(cnt){cnt--;}else{ I=*R;P[0]=P[1]=XPal[I? I:T[3]&0x0F]; P+=2; } R++;
			T+=4;
		}else{
			for(X=0;X<HAdjust;X++ ){
				P[0]=P[1]= XPal[BGColor];P+=2;
			}
		}
	}
    for(X=0;X<XE;X++,R+=8,P+=16,T+=4)
    {
      I=R[ 0];P[ 0]=P[ 1]=XPal[I? I:T[0]>>4];
      I=R[ 1];P[ 2]=P[ 3]=XPal[I? I:T[0]&0x0F];
      I=R[ 2];P[ 4]=P[ 5]=XPal[I? I:T[1]>>4];
      I=R[ 3];P[ 6]=P[ 7]=XPal[I? I:T[1]&0x0F];
      I=R[ 4];P[ 8]=P[ 9]=XPal[I? I:T[2]>>4];
      I=R[ 5];P[10]=P[11]=XPal[I? I:T[2]&0x0F];
      I=R[ 6];P[12]=P[13]=XPal[I? I:T[3]>>4];
      I=R[ 7];P[14]=P[15]=XPal[I? I:T[3]&0x0F];
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				P[0]=P[1]= XPal[BGColor];P+=2;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ I=*R;P[0]=P[1]=XPal[I? I:T[0]>>4];   cnt--; R++;P+=2; }
			if(cnt){ I=*R;P[0]=P[1]=XPal[I? I:T[0]&0x0F]; cnt--; R++;P+=2; }
			if(cnt){ I=*R;P[0]=P[1]=XPal[I? I:T[1]>>4];   cnt--; R++;P+=2; }
			if(cnt){ I=*R;P[0]=P[1]=XPal[I? I:T[1]&0x0F]; cnt--; R++;P+=2; }
			if(cnt){ I=*R;P[0]=P[1]=XPal[I? I:T[2]>>4];   cnt--; R++;P+=2; }
			if(cnt){ I=*R;P[0]=P[1]=XPal[I? I:T[2]&0x0F]; cnt--; R++;P+=2; }
			if(cnt){ I=*R;P[0]=P[1]=XPal[I? I:T[3]>>4];   cnt--; R++;P+=2; }
			if(cnt){ I=*R;P[0]=P[1]=XPal[I? I:T[3]&0x0F]; cnt--; R++;P+=2; }
		}
	}
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine6() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN6, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine6_x2(register byte Y)
{
  register pixel *P,*OP;
  register byte X,XE,*T,*R,C;
  register int cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,XPal[BGColor&0x03],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor&0x03],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<7)&ChrTabM&0x7FFF);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		if( HAdjust<0 ){
			cnt = -HAdjust*2;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C: T[0]>>6];       P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[0]>>4)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[0]>>2)&0x03]; P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[0]   )&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C: T[1]>>6];       P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[1]>>4)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[1]>>2)&0x03]; P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[1]   )&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C: T[2]>>6];       P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[2]>>4)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[2]>>2)&0x03]; P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[2]   )&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C: T[3]>>6      ]; P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[3]>>4)&0x03]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[3]>>2)&0x03]; P++; } ;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:(T[3]   )&0x03]; P++; } R++;
			T+=4;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor&0x03];P++;
			}
		}
	}
    for(X=0;X<XE;X++)
    {
      C=R[0];P[ 0]=XPal[C? C:(T[0]>>6)];
      C=R[0];P[ 1]=XPal[C? C:(T[0]>>4)&0x03];
      C=R[1];P[ 2]=XPal[C? C:(T[0]>>2)&0x03];
      C=R[1];P[ 3]=XPal[C? C: T[0]    &0x03];
      C=R[2];P[ 4]=XPal[C? C:(T[1]>>6)];
      C=R[2];P[ 5]=XPal[C? C:(T[1]>>4)&0x03];
      C=R[3];P[ 6]=XPal[C? C:(T[1]>>2)&0x03];
      C=R[3];P[ 7]=XPal[C? C: T[1]    &0x03];
      C=R[4];P[ 8]=XPal[C? C:(T[2]>>6)];
      C=R[4];P[ 9]=XPal[C? C:(T[2]>>4)&0x03];
      C=R[5];P[10]=XPal[C? C:(T[2]>>2)&0x03];
      C=R[5];P[11]=XPal[C? C: T[2]    &0x03];
      C=R[6];P[12]=XPal[C? C:(T[3]>>6)];
      C=R[6];P[13]=XPal[C? C:(T[3]>>4)&0x03];
      C=R[7];P[14]=XPal[C? C:(T[3]>>2)&0x03];
      C=R[7];P[15]=XPal[C? C: T[3]    &0x03];
      R+=8;P+=16;T+=4;
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust*2;X++ ){
				*P = XPal[BGColor&0x03];P++;
			}
		}else{
			cnt = 16 - HAdjust*2;
			if(cnt){ C=*R;*P=XPal[C? C: T[0]>>6      ]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[0]>>4)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[0]>>2)&0x03]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[0]   )&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C: T[1]>>6      ]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[1]>>4)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[1]>>2)&0x03]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[1]   )&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C: T[2]>>6      ]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[2]>>4)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[2]>>2)&0x03]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[2]   )&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C: T[3]>>6      ]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[3]>>4)&0x03]; cnt--; R++;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[3]>>2)&0x03]; cnt--;    ;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:(T[3]   )&0x03]; cnt--; R++;P++; }
		}
	}
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}
  
/** RefreshLine7() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN7, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine7_x2(register byte Y)
{
  register pixel *P,*OP;
  register byte C,X,XE,*T,*R;
  register int cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,XPal[BGColor],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,XPal[BGColor],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		if( HAdjust<0 ){
			cnt = -HAdjust*2;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[0]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[0]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[1]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[1]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[2]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[2]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[3]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[3]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[4]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[4]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[5]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[5]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[6]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[6]&0xF]; P++; } R++;
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[7]>>4 ]; P++; };
			if(cnt){cnt--;}else{ C=*R;*P=XPal[C? C:T[7]&0xF]; P++; } R++;
			T+= 8;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = XPal[BGColor];P++;
			}
		}
	}
    for(X=0;X<XE;X++)
    {
      C=R[0];P[ 0]=XPal[C? C:T[0]>>4];
      C=R[0];P[ 1]=XPal[C? C:T[0]   &0xF];
      C=R[1];P[ 2]=XPal[C? C:T[1]>>4];
      C=R[1];P[ 3]=XPal[C? C:T[1]   &0xF];
      C=R[2];P[ 4]=XPal[C? C:T[2]>>4];
      C=R[2];P[ 5]=XPal[C? C:T[2]   &0xF];
      C=R[3];P[ 6]=XPal[C? C:T[3]>>4];
      C=R[3];P[ 7]=XPal[C? C:T[3]   &0xF];
      C=R[4];P[ 8]=XPal[C? C:T[4]>>4];
      C=R[4];P[ 9]=XPal[C? C:T[4]   &0xF];
      C=R[5];P[10]=XPal[C? C:T[5]>>4];
      C=R[5];P[11]=XPal[C? C:T[5]   &0xF];
      C=R[6];P[12]=XPal[C? C:T[6]>>4];
      C=R[6];P[13]=XPal[C? C:T[6]   &0xF];
      C=R[7];P[14]=XPal[C? C:T[7]>>4];
      C=R[7];P[15]=XPal[C? C:T[7]   &0xF];
      R+=8;P+=16;T+=8;
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust*2;X++ ){
				*P = XPal[BGColor];P++;
			}
		}else{
			cnt = 16 - HAdjust*2;
			if(cnt){ C=*R;*P=XPal[C? C:T[0]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[0]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[1]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[1]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[2]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[2]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[3]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[3]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[4]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[4]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[5]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[5]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[6]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[6]&0xF]; cnt--;P++;R++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[7]>>4 ]; cnt--;P++; }
			if(cnt){ C=*R;*P=XPal[C? C:T[7]&0xF]; cnt--;P++;R++; }
		}
	}
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine8() *******************************************/
/** Refresh line Y (0..191/211) of SCREEN8, including       **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine8_x2(register byte Y)
{
  static byte SprToScr[16] =
  {
    0x00,0x02,0x10,0x12,0x80,0x82,0x90,0x92,
    0x49,0x4B,0x59,0x5B,0xC9,0xCB,0xD9,0xDB
  };
  register pixel *P,*OP;
  register byte C,X,XE,*T,*R;
  register int cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,BPal[VDP[7]],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BPal[VDP[7]],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

	if(HAdjust==0){
		XE=32;
	}else {
		XE=31;

		if( HAdjust<0 ){
			cnt = -HAdjust;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[0]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[1]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[2]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[3]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[4]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[5]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[6]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[7]]; P+=2; } R++;
			T+= 8;
		}else{
			for(X=0;X<HAdjust;X++ ){
				P[0]=P[1]= BPal[VDP[7]];P+=2;
			}
		}
	}
    for(X=0;X<XE;X++,T+=8,R+=8,P+=16)
    {
      C=R[0];P[ 0]=P[ 1]=BPal[C? SprToScr[C]:T[0]];
      C=R[1];P[ 2]=P[ 3]=BPal[C? SprToScr[C]:T[1]];
      C=R[2];P[ 4]=P[ 5]=BPal[C? SprToScr[C]:T[2]];
      C=R[3];P[ 6]=P[ 7]=BPal[C? SprToScr[C]:T[3]];
      C=R[4];P[ 8]=P[ 9]=BPal[C? SprToScr[C]:T[4]];
      C=R[5];P[10]=P[11]=BPal[C? SprToScr[C]:T[5]];
      C=R[6];P[12]=P[13]=BPal[C? SprToScr[C]:T[6]];
      C=R[7];P[14]=P[15]=BPal[C? SprToScr[C]:T[7]];
    }
	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<-HAdjust;X++ ){
				P[0]=P[1]= BPal[VDP[7]];P+=2;
			}
		}else{
			cnt = 8 - HAdjust;
			if(cnt){ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[0]]; cnt--;P+=2;R++; }
			if(cnt){ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[1]]; cnt--;P+=2;R++; }
			if(cnt){ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[2]]; cnt--;P+=2;R++; }
			if(cnt){ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[3]]; cnt--;P+=2;R++; }
			if(cnt){ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[4]]; cnt--;P+=2;R++; }
			if(cnt){ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[5]]; cnt--;P+=2;R++; }
			if(cnt){ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[6]]; cnt--;P+=2;R++; }
			if(cnt){ C=*R;P[0]=P[1]=BPal[C? SprToScr[C]:T[7]]; cnt--;P+=2;R++; }
		}
	}
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine10() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN10/11, including   **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine10_x2(register byte Y)
{
  register pixel *P,*OP;
  register byte C,X,XE,*T,*R;
  register int J,K,cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,BPal[VDP[7]],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BPal[VDP[7]],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

	if(HAdjust==0){
		XE=63;
		/* Draw first 4 pixels */
		C=R[0];P[ 0]=P[ 1]=C? XPal[C]:BPal[VDP[7]];
		C=R[1];P[ 2]=P[ 3]=C? XPal[C]:BPal[VDP[7]];
		C=R[2];P[ 4]=P[ 5]=C? XPal[C]:BPal[VDP[7]];
		C=R[3];P[ 6]=P[ 7]=C? XPal[C]:BPal[VDP[7]];
		R+=4;P+=8;
	}else {
		XE=62;
		if( HAdjust<0 ){

			cnt = -HAdjust;
		    /* Draw first 4 pixels */
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:BPal[VDP[7]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:BPal[VDP[7]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:BPal[VDP[7]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:BPal[VDP[7]]; P+=2; } R++;

			K=(T[0]&0x07)|((T[1]&0x07)<<3);
			if(K&0x20) K-=64;
			J=(T[2]&0x07)|((T[3]&0x07)<<3);
			if(J&0x20) J-=64;

			if(cnt){cnt--;}else{ C=*R;Y=T[0]>>3;P[0]=P[1]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;Y=T[1]>>3;P[0]=P[1]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;Y=T[2]>>3;P[0]=P[1]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;Y=T[3]>>3;P[0]=P[1]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); P+=2; } R++;
			T+=4;
		}else{
			for(X=0;X<HAdjust;X++ ){
				P[0]=P[1]= BPal[VDP[7]];P+=2;
			}
			C=R[0];P[0]=P[1]=C? XPal[C]:BPal[VDP[7]];
			C=R[1];P[2]=P[3]=C? XPal[C]:BPal[VDP[7]];
			C=R[2];P[4]=P[5]=C? XPal[C]:BPal[VDP[7]];
			C=R[3];P[6]=P[7]=C? XPal[C]:BPal[VDP[7]];
			R+=4;P+=8;
		}
	}
    for(X=0;X<XE;X++,T+=4,R+=4,P+=8)
    {
      K=(T[0]&0x07)|((T[1]&0x07)<<3);
      if(K&0x20) K-=64;
      J=(T[2]&0x07)|((T[3]&0x07)<<3);
      if(J&0x20) J-=64;

      C=R[0];Y=T[0]>>3;P[ 0]=P[ 1]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
      C=R[1];Y=T[1]>>3;P[ 2]=P[ 3]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
      C=R[2];Y=T[2]>>3;P[ 4]=P[ 5]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
      C=R[3];Y=T[3]>>3;P[ 6]=P[ 7]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K);
    }

	if( HAdjust!= 0) {
		if( HAdjust<0 ){
			for(X=0;X<HAdjust;X++ ){
				P[0]=P[1] = BPal[VDP[7]];P+=2;
			}
		}else{
			cnt = 8 - HAdjust;
			for(X=0;X<2;X++){
				K=(T[0]&0x07)|((T[1]&0x07)<<3);
				if(K&0x20) K-=64;
				J=(T[2]&0x07)|((T[3]&0x07)<<3);
				if(J&0x20) J-=64;
				if(cnt){ C=R[0];Y=T[0]>>3;P[0]=P[1]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); cnt--; } else break;
				if(cnt){ C=R[1];Y=T[1]>>3;P[2]=P[3]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); cnt--; } else break;
				if(cnt){ C=R[2];Y=T[2]>>3;P[4]=P[5]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); cnt--; } else break;
				if(cnt){ C=R[3];Y=T[3]>>3;P[6]=P[7]=C? XPal[C]:Y&1? XPal[Y>>1]:YJKColor(Y,J,K); cnt--; } else break;
				T+=4;R+=4;P+=8;
			}
		}
	}
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

/** RefreshLine12() ******************************************/
/** Refresh line Y (0..191/211) of SCREEN12, including      **/
/** sprites in this line.                                   **/
/*************************************************************/
void RefreshLine12_x2(register byte Y)
{
  register pixel *P,*OP;
  register byte C,X,XE,*T,*R;
  register int J,K,cnt;
  byte ZBuf[304];

  OP=P=RefreshBorder(Y,BPal[VDP[7]],512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BPal[VDP[7]],512);
  else
  {
    ColorSprites(Y,ZBuf);
    R=ZBuf+32;
    T=ChrTab+(((int)(Y+VScroll)<<8)&ChrTabM&0xFFFF);

    if(HScroll512&&(HScroll>255)) T=(byte *)((int)T^0x10000);
    T+=HScroll&0xFC;

	if(HAdjust==0){
		XE=63;
	    /* Draw first 4 pixels */
	    C=R[0];P[ 0]=P[ 1]=C? XPal[C]:BPal[VDP[7]];
	    C=R[1];P[ 2]=P[ 3]=C? XPal[C]:BPal[VDP[7]];
	    C=R[2];P[ 4]=P[ 5]=C? XPal[C]:BPal[VDP[7]];
	    C=R[3];P[ 6]=P[ 7]=C? XPal[C]:BPal[VDP[7]];
	    R+=4;P+=8;
	}else {
		XE=62;

		if( HAdjust<0 ){
			cnt = -HAdjust;
			/* Draw first 4 pixels */
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:BPal[VDP[7]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:BPal[VDP[7]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:BPal[VDP[7]]; P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:BPal[VDP[7]]; P+=2; } R++;

			K=(T[0]&0x07)|((T[1]&0x07)<<3);
			if(K&0x20) K-=64;
			J=(T[2]&0x07)|((T[3]&0x07)<<3);
			if(J&0x20) J-=64;

			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:YJKColor(T[0]>>3,J,K); P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:YJKColor(T[1]>>3,J,K); P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:YJKColor(T[2]>>3,J,K); P+=2; } R++;
			if(cnt){cnt--;}else{ C=*R;P[0]=P[1]=C? XPal[C]:YJKColor(T[3]>>3,J,K); P+=2; } R++;
			T++;
		}else{
			for(X=0;X<HAdjust;X++ ){
				*P = BPal[VDP[7]];P++;
			}
			/* Draw first 4 pixels */
			C=R[0];P[0]=C? XPal[C]:BPal[VDP[7]];
			C=R[1];P[1]=C? XPal[C]:BPal[VDP[7]];
			C=R[2];P[2]=C? XPal[C]:BPal[VDP[7]];
			C=R[3];P[3]=C? XPal[C]:BPal[VDP[7]];
			R+=4;P+=4;
		}
	}
    for(X=1;X<XE;X++,T+=4,R+=4,P+=8)
    {
      K=(T[0]&0x07)|((T[1]&0x07)<<3);
      if(K&0x20) K-=64;
      J=(T[2]&0x07)|((T[3]&0x07)<<3);
      if(J&0x20) J-=64;

      C=R[0];P[ 0]=P[ 1]=C? XPal[C]:YJKColor(T[0]>>3,J,K);
      C=R[1];P[ 2]=P[ 3]=C? XPal[C]:YJKColor(T[1]>>3,J,K);
      C=R[2];P[ 4]=P[ 5]=C? XPal[C]:YJKColor(T[2]>>3,J,K);
      C=R[3];P[ 6]=P[ 7]=C? XPal[C]:YJKColor(T[3]>>3,J,K);
    }
	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}


/** RefreshLineTx80() ****************************************/
/** Refresh line Y (0..191/211) of TEXT80.                  **/
/*************************************************************/
void RefreshLineTx80_x2(register byte Y)
{
  register pixel *P,*OP,FC,BC;
  register byte X,M,*T,*C,*G;

  BC=XPal[BGColor];
  OP=P=RefreshBorder(Y,BC,512,528);
  if(!P) return;

  if(!ScreenON) ClearLine(P,BC,512);
  else
  {
    P[0]=P[1]=P[2]=P[3]=P[4]=P[5]=P[6]=P[7]=P[8]=BC;
    G=(UseFont&&FontBuf? FontBuf:ChrGen)+((Y+VScroll)&0x07);
    T=ChrTab+80*(Y>>3);
    C=ColTab+10*(Y>>3);
    P+=9;

    for(X=0,M=0x00;X<80;X++,T++,P+=6)
    {
      if(!(X&0x07)) M=*C++;
      if(M&0x80) { FC=XPal[XFGColor];BC=XPal[XBGColor]; }
      else       { FC=XPal[ FGColor];BC=XPal[BGColor]; }
      M<<=1;
      Y=*(G+((int)*T<<3));
      P[0]=Y&0x80? FC:BC;
      P[1]=Y&0x40? FC:BC;
      P[2]=Y&0x20? FC:BC;
      P[3]=Y&0x10? FC:BC;
      P[4]=Y&0x08? FC:BC;
      P[5]=Y&0x04? FC:BC;
    }

    P[0]=P[1]=P[2]=P[3]=P[4]=P[5]=P[6]=XPal[BGColor];

	if( !UseInterless ) {
		memcpy( OP+544, OP, sizeof(pixel) * 512 ); 
	}
  }
}

void (*RefreshLineList[3][MAXSCREEN+2])(byte Y) = 
{
 {
  RefreshLine0,   /* SCR 0:  TEXT 40x24  */
  RefreshLine1,   /* SCR 1:  TEXT 32x24  */
  RefreshLine2,   /* SCR 2:  BLK 256x192 */
  RefreshLine3,   /* SCR 3:  64x48x16    */
  RefreshLine4,   /* SCR 4:  BLK 256x192 */
  RefreshLine5,   /* SCR 5:  256x192x16  */
  RefreshLine6,   /* SCR 6:  512x192x4   */
  RefreshLine7,   /* SCR 7:  512x192x16  */
  RefreshLine8,   /* SCR 8:  256x192x256 */
  0,              /* SCR 9:  NONE        */
  RefreshLine10,  /* SCR 10: YAE 256x192 */
  RefreshLine10,  /* SCR 11: YAE 256x192 */
  RefreshLine12,  /* SCR 12: YJK 256x192 */
  RefreshLineTx80 /* SCR 0:  TEXT 80x24  */
},
{
  RefreshLine0,   /* SCR 0:  TEXT 40x24  */
  RefreshLine1,   /* SCR 1:  TEXT 32x24  */
  RefreshLine2,   /* SCR 2:  BLK 256x192 */
  RefreshLine3,   /* SCR 3:  64x48x16    */
  RefreshLine4,   /* SCR 4:  BLK 256x192 */
  RefreshLine5,   /* SCR 5:  256x192x16  */
  RefreshLine6W,  /* SCR 6:  512x192x4   */
  RefreshLine7W,  /* SCR 7:  512x192x16  */
  RefreshLine8,   /* SCR 8:  256x192x256 */
  0,              /* SCR 9:  NONE        */
  RefreshLine10,  /* SCR 10: YAE 256x192 */
  RefreshLine10,  /* SCR 11: YAE 256x192 */
  RefreshLine12,  /* SCR 12: YJK 256x192 */
  RefreshLineTx80W/* SCR 0:  TEXT 80x24  */
},
{
  RefreshLine0_x2,   /* SCR 0:  TEXT 40x24  */
  RefreshLine1_x2,   /* SCR 1:  TEXT 32x24  */
  RefreshLine2_x2,   /* SCR 2:  BLK 256x192 */
  RefreshLine3_x2,   /* SCR 3:  64x48x16    */
  RefreshLine4_x2,   /* SCR 4:  BLK 256x192 */
  RefreshLine5_x2,   /* SCR 5:  256x192x16  */
  RefreshLine6_x2,   /* SCR 6:  512x192x4   */
  RefreshLine7_x2,   /* SCR 7:  512x192x16  */
  RefreshLine8_x2,   /* SCR 8:  256x192x256 */
  0,                 /* SCR 9:  NONE        */
  RefreshLine10_x2,  /* SCR 10: YAE 256x192 */
  RefreshLine10_x2,  /* SCR 11: YAE 256x192 */
  RefreshLine12_x2,  /* SCR 12: YJK 256x192 */
  RefreshLineTx80_x2 /* SCR 0:  TEXT 80x24  */
 }
};
void (*RefreshLine[MAXSCREEN+2])(byte Y);

void setRefreshLine(int flg){
	int i;
	for(i=0;i<MAXSCREEN+2;i++){
		RefreshLine[i] = RefreshLineList[flg][i];
	}
}
