

#include <stdio.h>
#include <string.h>

#include "defs.h"
#include "cpu.h"
#include "cpuregs.h"
#include "hw.h"
#include "regs.h"
#include "lcd.h"
#include "rtc.h"
#include "mem.h"
#include "sound.h"

#include "stdfnc.h"

#ifdef IS_LITTLE_ENDIAN
#define LIL(x) (x)
#else
#define LIL(x) ((x<<24)|((x&0xff00)<<8)|((x>>8)&0xff00)|(x>>24))
#endif

#define I1(s1,s2,s3,s4, p) { 1, {s1,s2,s3,s4} , p }
#define I2(s1,s2,s3,s4, p) { 2, {s1,s2,s3,s4} , p }
#define I4(s1,s2,s3,s4, p) { 4, {s1,s2,s3,s4} , p }
#define R(r) I1(#r, &R_##r)
#define NOSAVE { -1, { '\0','\0','\0','\0'} , 0 }
#define END { 0, { '\0','\0','\0','\0'} , 0 }

struct svar
{
	int len;
	char key[4];
	void *ptr;
};

static int ver;
static int sramblock, iramblock, vramblock;
static int hramofs, hiofs, palofs, oamofs, wavofs;

struct svar svars[] = 
{
	I4('G','b','S','s', &ver),
	
	I2('P','C',' ',' ', &PC),
	I2('S','P',' ',' ', &SP),
	I2('B','C',' ',' ', &BC),
	I2('D','E',' ',' ', &DE),
	I2('H','L',' ',' ', &HL),
	I2('A','F',' ',' ', &AF),
	
	I4('I','M','E',' ', &cpu.ime),
	I4('i','m','a',' ', &cpu.ima),
	I4('s','p','d',' ', &cpu.speed),
	I4('h','a','l','t', &cpu.halt),
	I4('d','i','v',' ', &cpu.div),
	I4('t','i','m',' ', &cpu.tim),
	I4('l','c','d','c', &cpu.lcdc),
	I4('s','n','d',' ', &cpu.snd),
	
	I1('i','n','t','s', &hw.ilines),
	I1('p','a','d',' ', &hw.pad),
	I4('c','g','b',' ', &hw.cgb),
	I4('g','b','a',' ', &hw.gba),
	
	I4('m','b','c','m', &mbc.model),
	I4('r','o','m','b', &mbc.rombank),
	I4('r','a','m','b', &mbc.rambank),
	I4('e','n','a','b', &mbc.enableram),
	I4('b','a','t','t', &mbc.batt),
	
	I4('r','t','c','R', &rtc.sel),
	I4('r','t','c','L', &rtc.latch),
	I4('r','t','c','C', &rtc.carry),
	I4('r','t','c','S', &rtc.stop),
	I4('r','t','c','d', &rtc.d),
	I4('r','t','c','h', &rtc.h),
	I4('r','t','c','m', &rtc.m),
	I4('r','t','c','s', &rtc.s),
	I4('r','t','c','t', &rtc.t),
	I1('r','t','R','8', &rtc.regs[0]),
	I1('r','t','R','9', &rtc.regs[1]),
	I1('r','t','R','A', &rtc.regs[2]),
	I1('r','t','R','B', &rtc.regs[3]),
	I1('r','t','R','C', &rtc.regs[4]),

	I4('S','1','o','n', &snd.ch[0].on),
	I4('S','1','p',' ', &snd.ch[0].pos),
	I4('S','1','c',' ', &snd.ch[0].cnt),
	I4('S','1','e','c', &snd.ch[0].encnt),
	I4('S','1','s','c', &snd.ch[0].swcnt),
	I4('S','1','s','f', &snd.ch[0].swfreq),

	I4('S','2','o','n', &snd.ch[1].on),
	I4('S','2','p',' ', &snd.ch[1].pos),
	I4('S','2','c',' ', &snd.ch[1].cnt),
	I4('S','2','e','c', &snd.ch[1].encnt),
	
	I4('S','3','o','n', &snd.ch[2].on),
	I4('S','3','p',' ', &snd.ch[2].pos),
	I4('S','3','c',' ', &snd.ch[2].cnt),
	
	I4('S','4','o','n', &snd.ch[3].on),
	I4('S','4','p',' ', &snd.ch[3].pos),
	I4('S','4','c',' ', &snd.ch[3].cnt),
	I4('S','4','e','c', &snd.ch[3].encnt),
	
	I4('h','d','m','a', &hw.hdma),
	
	I4('s','r','a','m', &sramblock),
	I4('i','r','a','m', &iramblock),
	I4('v','r','a','m', &vramblock),
	I4('h','i',' ',' ', &hiofs),
	I4('p','a','l',' ', &palofs),
	I4('o','a','m',' ', &oamofs),
	I4('w','a','v',' ', &wavofs),
	
	/* NOSAVE is a special code to prevent the rest of the table
	 * from being saved, used to support old stuff for backwards
	 * compatibility... */
	NOSAVE,

	/* the following are obsolete as of 0x104 */
	
	I4('h','r','a','m', &hramofs),
	
	I1('P','1',' ',' ',&R_P1  ),
	I1('S','B',' ',' ',&R_SB  ),
	I1('S','C',' ',' ',&R_SC  ),
	I1('D','I','V',' ',&R_DIV ),
	I1('T','I','M','A',&R_TIMA),
	I1('T','M','A',' ',&R_TMA ),
	I1('T','A','C',' ',&R_TAC ),
	I1('I','E',' ',' ',&R_IE  ),
	I1('I','F',' ',' ',&R_IF  ),
	I1('L','C','D','C',&R_LCDC),
	I1('S','T','A','T',&R_STAT),
	I1('L','Y',' ',' ',&R_LY  ),
	I1('L','Y','C',' ',&R_LYC ),
	I1('S','C','X',' ',&R_SCX ),
	I1('S','C','Y',' ',&R_SCY ),
	I1('W','X',' ',' ',&R_WX  ),
	I1('W','Y',' ',' ',&R_WY  ),
	I1('B','G','P',' ',&R_BGP ),
	I1('O','B','P','0',&R_OBP0),
	I1('O','B','P','1',&R_OBP1),
	I1('D','M','A',' ',&R_DMA ),
                 
	I1('V','B','K',' ',&R_VBK ),
	I1('S','V','B','K',&R_SVBK),
	I1('K','E','Y','1',&R_KEY1),
	I1('B','C','P','S',&R_BCPS),
	I1('B','C','P','D',&R_BCPD),
	I1('O','C','P','S',&R_OCPS),
	I1('O','C','P','D',&R_OCPD),
                 
	I1('N','R','1','0',&R_NR10),
	I1('N','R','1','1',&R_NR11),
	I1('N','R','1','2',&R_NR12),
	I1('N','R','1','3',&R_NR13),
	I1('N','R','1','4',&R_NR14),
	I1('N','R','2','1',&R_NR21),
	I1('N','R','2','2',&R_NR22),
	I1('N','R','2','3',&R_NR23),
	I1('N','R','2','4',&R_NR24),
	I1('N','R','3','0',&R_NR30),
	I1('N','R','3','1',&R_NR31),
	I1('N','R','3','2',&R_NR32),
	I1('N','R','3','3',&R_NR33),
	I1('N','R','3','4',&R_NR34),
	I1('N','R','4','1',&R_NR41),
	I1('N','R','4','2',&R_NR42),
	I1('N','R','4','3',&R_NR43),
	I1('N','R','4','4',&R_NR44),
	I1('N','R','5','0',&R_NR50),
	I1('N','R','5','1',&R_NR51),
	I1('N','R','5','2',&R_NR52),

	I1('D','M','A','1', &R_HDMA1),
	I1('D','M','A','2', &R_HDMA2),
	I1('D','M','A','3', &R_HDMA3),
	I1('D','M','A','4', &R_HDMA4),
	I1('D','M','A','5', &R_HDMA5),
	
	END
};

void loadstate(FILE *f)
{
	int i, j;
	byte buf[4096];
	un32 (*header)[2] = (un32 (*)[2])buf;
	un32 d;
	int irl = hw.cgb ? 8 : 2;
	int vrl = hw.cgb ? 4 : 2;
	int srl = mbc.ramsize << 1;

	ver = hramofs = hiofs = palofs = oamofs = wavofs = 0;

	fseek(f, 0, SEEK_SET);
	fread(buf, 4096, 1, f);
	
	for (j = 0; header[j][0]; j++)
	{
		for (i = 0; svars[i].ptr; i++)
		{
			if (header[j][0] != *(un32 *)svars[i].key)
				continue;
			d = LIL(header[j][1]);
			switch (svars[i].len)
			{
			case 1:
				*(byte *)svars[i].ptr = d;
				break;
			case 2:
				*(un16 *)svars[i].ptr = d;
				break;
			case 4:
				*(un32 *)svars[i].ptr = d;
				break;
			}
			break;
		}
	}

	/* obsolete as of version 0x104 */
	if (hramofs) memcpy(ram.hi+128, buf+hramofs, 127);
	
	if (hiofs) memcpy(ram.hi, buf+hiofs, sizeof ram.hi);
	if (palofs) memcpy(lcd.pal, buf+palofs, sizeof lcd.pal);
	if (oamofs) memcpy(lcd.oam.mem, buf+oamofs, sizeof lcd.oam);

	if (wavofs) memcpy(snd.wave, buf+wavofs, sizeof snd.wave);
	else memcpy(snd.wave, ram.hi+0x30, 16); /* patch data from older files */

	fseek(f, iramblock<<12, SEEK_SET);
	fread(ram.ibank, 4096, irl, f);
	
	fseek(f, vramblock<<12, SEEK_SET);
	fread(lcd.vbank, 4096, vrl, f);
	
	fseek(f, sramblock<<12, SEEK_SET);
	fread(ram.sbank, 4096, srl, f);
}

void savestate(FILE *f)
{
	int i;
	byte buf[4096];
	un32 (*header)[2] = (un32 (*)[2])buf;
	un32 d = 0;
	int irl = hw.cgb ? 8 : 2;
	int vrl = hw.cgb ? 4 : 2;
	int srl = mbc.ramsize << 1;

	ver = 0x105;
	iramblock = 1;
	vramblock = 1+irl;
	sramblock = 1+irl+vrl;
	wavofs = 4096 - 784;
	hiofs = 4096 - 768;
	palofs = 4096 - 512;
	oamofs = 4096 - 256;
	memset(buf, 0, sizeof buf);

	for (i = 0; svars[i].len > 0; i++)
	{
		header[i][0] = *(un32 *)svars[i].key;
		switch (svars[i].len)
		{
		case 1:
			d = *(byte *)svars[i].ptr;
			break;
		case 2:
			d = *(un16 *)svars[i].ptr;
			break;
		case 4:
			d = *(un32 *)svars[i].ptr;
			break;
		}
		header[i][1] = LIL(d);
	}
	header[i][0] = header[i][1] = 0;

	memcpy(buf+hiofs, ram.hi, sizeof ram.hi);
	memcpy(buf+palofs, lcd.pal, sizeof lcd.pal);
	memcpy(buf+oamofs, lcd.oam.mem, sizeof lcd.oam);
	memcpy(buf+wavofs, snd.wave, sizeof snd.wave);

	fseek(f, 0, SEEK_SET);
	fwrite(buf, 4096, 1, f);
	
	fseek(f, iramblock<<12, SEEK_SET);
	fwrite(ram.ibank, 4096, irl, f);
	
	fseek(f, vramblock<<12, SEEK_SET);
	fwrite(lcd.vbank, 4096, vrl, f);
	
	fseek(f, sramblock<<12, SEEK_SET);
	fwrite(ram.sbank, 4096, srl, f);
}



















