/* Keyboard library functions */
/* H. Hanemaayer (hhanemaa@cs.ruu.nl) */

/*
 * Keyboard I/O based on showkey.c from kbd-0.84 package.
 *
 * This is an initial version, it isn't very safe yet since it only catches
 * sigsegv.
 */

/* DOS port by Mihai Cartoaje ( moisemih@ift.ulaval.ca ).
 */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#include <pc.h>

#define	port_in	inportb
#define	port_out	outportb

#include "vgakeybo.h"
#include <dpmi.h>
#include <go32.h>

static void (*__keyboard_eventhandler) (int, int);

/* vga.c needs to check that: */
int kbd_fd = -1;
static int ctrl_c_state_c, ctrl_c_state_control;
static int alt_state, functionkey_state;
static int translatemode = 0;

static unsigned char state[127];        /* This should be enough for most keyboards*/

static void default_handler(int, int);


static _go32_dpmi_seginfo wrapper;
static __dpmi_paddr oldHandler, newHandler;

#define DJGPPKEYBOARDBUFFERSIZE 256

char 	djgpp_buf[DJGPPKEYBOARDBUFFERSIZE];
int 	djgpp_bufferstart= 0, djgpp_bufferend = 0;
/* Must do our own interrupt handling. Luckily, DOS allows this. */

/* Temporary */
extern volatile int joyinterrupt;

void kbdintr ( void ) 
{
	int tmp;
        
	tmp = port_in ( 0x60 );

	/* Insert keystroke into a queue awaiting processing */

	djgpp_buf [ djgpp_bufferend ++ ] = tmp;
	djgpp_bufferend &= DJGPPKEYBOARDBUFFERSIZE - 1;
	if ( djgpp_bufferend == djgpp_bufferstart ) {
		djgpp_bufferstart ++;
		djgpp_bufferstart &= DJGPPKEYBOARDBUFFERSIZE - 1;
	} 
        port_out ( 0x20, 0x20 );        /* Acknowledge the interrupt */
}

int lock_memory(const void *ptr, unsigned long bytes) {
  unsigned long base;

  if(__dpmi_get_segment_base_address(_my_ds(),&base)!=-1) {
	__dpmi_meminfo mem;
	mem.handle=0;
	mem.size=bytes;
	mem.address=(unsigned long)((char *)ptr+base);
	if(__dpmi_lock_linear_region(&mem)!=-1)
	  return 1;
  }

  return -1;
}


int keyboard_init_return_fd(void)
{

    /* Install default keyboard handler. */
    __keyboard_eventhandler = default_handler;

    kbd_fd = 1;             /* So that vga.c can verify keyboard has been inited*/

    wrapper.pm_offset=(int)kbdintr;
    wrapper.pm_selector=_my_cs();
    _go32_dpmi_allocate_iret_wrapper(&wrapper);
    newHandler.offset32=wrapper.pm_offset;
    newHandler.selector=wrapper.pm_selector;

    lock_memory(kbdintr,(long)keyboard_init_return_fd-(long)kbdintr);

    __dpmi_get_and_disable_virtual_interrupt_state();        
    __dpmi_get_protected_mode_interrupt_vector(9,&oldHandler);
    __dpmi_set_protected_mode_interrupt_vector(9,&newHandler);
    __dpmi_get_and_enable_virtual_interrupt_state();

    keyboard_clearstate();

    return kbd_fd;		/* OK, return fd. */
}

/* Old compatible init function. */

int keyboard_init(void)
{
    if (keyboard_init_return_fd() == -1)
	return -1;
    else
	return 0;
}

void keyboard_close(void)
{
    if (kbd_fd < 0)
	return;

    __dpmi_get_and_disable_virtual_interrupt_state();
    __dpmi_set_protected_mode_interrupt_vector(9,&oldHandler);
    __dpmi_get_and_enable_virtual_interrupt_state();

    _go32_dpmi_free_iret_wrapper(&wrapper);

    kbd_fd = -1;
}


/* For now, we assume there's no console switching. */
/* (Actually, there won't be any unless we catch the console switching */
/* keys). */


#define KBDREADBUFFERSIZE 32

static int keyboard_getevents(int wait)
{
/* Read keyboard device, and handle events. */
/* If wait == 1, process at least one event and return. */
/* If wait == 0, handle all accumulated events; return 0 if no events */
/* were handled, 1 otherwise. */
/* Wait mode doesn't seem to work very well; the keyboard repeat delay is */
/* present. I don't understand fcntl. */
    static unsigned char buf[KBDREADBUFFERSIZE];

    int bytesread, i;
    int eventhandled;

    eventhandled = 0;

  again:

	waiting_loop:
        if (( bytesread = djgpp_bufferend - djgpp_bufferstart )) {
		int i;
		if ( bytesread < 0 ) bytesread += DJGPPKEYBOARDBUFFERSIZE;
		if ( bytesread > KBDREADBUFFERSIZE ) bytesread = KBDREADBUFFERSIZE;
		for ( i = 0; i < bytesread; i ++ ) {
			buf [ i ] = djgpp_buf [ djgpp_bufferstart ++ ];
			djgpp_bufferstart &= DJGPPKEYBOARDBUFFERSIZE - 1;
		}
	} else if ( wait == 1 ) goto waiting_loop; 

    if (wait == 0 && bytesread < 1)
	return eventhandled;

    if (bytesread >= 1)
	eventhandled = 1;

    for (i = 0; i < bytesread; i++) {
	/* Check for ctrl-c. */
	if ((buf[i] & 0x7f) == SCANCODE_C)
        {
	    if (buf[i] & 0x80)
		ctrl_c_state_c = 0;
	    else
		ctrl_c_state_c = 1;
        }
	if ((buf[i] & 0x7f) == SCANCODE_LEFTCONTROL
	    || (buf[i] & 0x7f) == SCANCODE_RIGHTCONTROL)
        {
	    if (buf[i] & 0x80)
		ctrl_c_state_control = 0;
	    else
                ctrl_c_state_control = 1;
        }
        if ((buf[i] & 0x7f) == SCANCODE_LEFTALT
	    || (buf[i] & 0x7f) == SCANCODE_RIGHTALT)
        {
	    if (buf[i] & 0x80)
		alt_state = 0;
	    else
		alt_state = 1;
        }
	if ((buf[i] & 0x7f) >= SCANCODE_F1
	    && (buf[i] & 0x7f) <= SCANCODE_F10)
        {
	    if (buf[i] & 0x80)
		functionkey_state &= ~(1 << ((buf[i] & 0x7f)
					     - SCANCODE_F1));
	    else
		functionkey_state |= 1 << ((buf[i] & 0x7f)
					   - SCANCODE_F1);
        }
	if (ctrl_c_state_control && ctrl_c_state_c
	    && !(translatemode & DONT_CATCH_CTRLC))
	    raise(SIGINT);

	__keyboard_eventhandler(buf[i] & 0x7f,
		    (buf[i] & 0x80) ? KEY_EVENTRELEASE : KEY_EVENTPRESS);
    }

    /* Handle other events that have accumulated. */
    goto again;
}

int keyboard_update(void)
{
    return keyboard_getevents(0);	/* Don't wait. */
}

void keyboard_waitforupdate(void)
{
    keyboard_getevents(1);	/* Wait for event. */
    return;
}

void keyboard_seteventhandler(void (*handler) (int, int))
{
    __keyboard_eventhandler = handler;
}



/* Default event handler. */

void keyboard_setdefaulteventhandler(void)
{
    __keyboard_eventhandler = default_handler;
}

static int checkscancode(int scancode)
{
    if ( scancode < 0 || scancode >= 127 )
    {
     return 1;
    }
    return 0;
}

static void default_handler(int scancode, int newstate)
{
    if (checkscancode(scancode))
	return;
    if (translatemode & TRANSLATE_CURSORKEYS)
	/* Map cursor key block to keypad cursor keys. */
	switch (scancode) {
	case SCANCODE_CURSORBLOCKUP:
	    scancode = SCANCODE_CURSORUP;
	    break;
	case SCANCODE_CURSORBLOCKLEFT:
	    scancode = SCANCODE_CURSORLEFT;
	    break;
	case SCANCODE_CURSORBLOCKRIGHT:
	    scancode = SCANCODE_CURSORRIGHT;
	    break;
	case SCANCODE_CURSORBLOCKDOWN:
	    scancode = SCANCODE_CURSORDOWN;
	    break;
	}
    if (translatemode & TRANSLATE_DIAGONAL) {
	/* Translate diagonal keypad keys to two keypad cursor keys. */
	switch (scancode) {
	case SCANCODE_CURSORUPLEFT:
	    state[SCANCODE_CURSORUP] = newstate;
	    state[SCANCODE_CURSORLEFT] = newstate;
	    return;
	case SCANCODE_CURSORUPRIGHT:
	    state[SCANCODE_CURSORUP] = newstate;
	    state[SCANCODE_CURSORRIGHT] = newstate;
	    return;
	case SCANCODE_CURSORDOWNLEFT:
	    state[SCANCODE_CURSORDOWN] = newstate;
	    state[SCANCODE_CURSORLEFT] = newstate;
	    return;
	case SCANCODE_CURSORDOWNRIGHT:
	    state[SCANCODE_CURSORDOWN] = newstate;
	    state[SCANCODE_CURSORRIGHT] = newstate;
	    return;
	}
    }
    if ((translatemode & TRANSLATE_KEYPADENTER) && scancode ==
	SCANCODE_KEYPADENTER)
	scancode = SCANCODE_ENTER;

    state[scancode] = newstate;
}

void keyboard_clearstate(void)
{
    memset ( state, 0, 127 );

    ctrl_c_state_c = 0;
    ctrl_c_state_control = 0;
    alt_state = 0;
    functionkey_state = 0;
}

int keyboard_keypressed(int scancode)
{
    if (checkscancode(scancode))
	return 0;
    return state[scancode];
}

char *keyboard_getstate(void)
{
    return state;
}

void keyboard_translatekeys(int mode)
{
    translatemode = mode;
}

