/* This code is based on Marat's "SndMSDOS.c", so here's the banner from the
   original file. */
/** EMULib Emulation Library *********************************/
/**                                                         **/
/**                         SndMSDOS.c                      **/
/**                                                         **/
/** This file contains standard sound generation routines   **/
/** for MSDOS using SoundBlaster and Adlib.                 **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1996-1999                 **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include <sys/farptr.h>
#include <ctype.h>
#include <pc.h>
#include <dos.h>
#include <dpmi.h>
#include <go32.h>

#include "../../driver.h"

#include "main.h"
#include "sb.h"
#include "joystick.h"

int dosmemselector;

/* Protected mode interrupt data. */
_go32_dpmi_seginfo Pa;
_go32_dpmi_seginfo PaBack;

/* Real mode interrupt data. */
_go32_dpmi_registers Pa2regs;
_go32_dpmi_seginfo Pa2;
_go32_dpmi_seginfo PaBack2;


static uint8 *WaveBuffer=0;
static int format;
static int frags,fragsize,fragtotal;
static volatile int WritePtr,ReadPtr;

static int SBPort,SBIRQ,VirtIRQ,SBDMA,SBDMA16;

static int PICMask[2];

static unsigned long Bluffer=0;
static int32 Segment=0;
static int IRQOld=0;
static void IRQHandler(_go32_dpmi_registers *r);

static int WriteDSP(int V);        /* Write value into DSP   */

/** WriteDSP() ***********************************************/
/** Write a value into SoundBlaster. Returns 0 if failed,   **/
/** 1 otherwise.                                            **/
/*************************************************************/
static int WriteDSP(register int V)
{
  int J;

  for(J=0;J<100000;J++)
    if(!(inp(SBPort+0x0C)&0x80))
    { 
     outp(SBPort+0x0C,V);
     return(1); 
    }
  return(0);
}

static int ReadDSP(uint8 *b)
{
  int J;

  for(J=100000;J;J--)
    if(inp(SBPort+0x0E)&0x80)
    { 
     *b=inp(SBPort+0x0A);
     return(1); 
    }
  return(0);
}

/** InitSB() *************************************************/
/** Initialize SoundBlaster for wave synthesis via DMA.     **/
/*************************************************************/
int InitSB(int Rate, int bittage)
{
  static int PagePort[8] = { 0x87,0x83,0x81,0x82,-1,0x8B,0x89,0x8A };
  uint16 DSPV;
  char *P;
  long J;
  uint8 tocheck;

  puts("Initializing Sound Blaster...");
  Segment=0;
  IRQOld=0;
  PICMask[0]=PICMask[1]=-1;

  format=bittage?1:0;

  frags=8;

  if(Rate<=11025)
   fragsize=1<<5;
  else if(Rate<=22050)
   fragsize=1<<6;
  else
   fragsize=1<<7;

  fragtotal=frags*fragsize;

  WaveBuffer=malloc(fragtotal<<format);

  if(format)
   memset(WaveBuffer,0,fragtotal*2);
  else
   memset(WaveBuffer,128,fragtotal);

  WritePtr=ReadPtr=0;

  /* If Rate is out of range, fall out */
  if((Rate<8192)||(Rate>65535))
  {
   printf(" Invalid sample rate: %d\n",Rate);
   return 0;
  }


  /* Get SoundBlaster Port#,IRQ#,DMA# */
  tocheck=0;
  if((P=getenv("BLASTER")))
    while(*P)
      switch(toupper(*P++))
      {
        case 'A': tocheck|=1;SBPort=strtol(P,0,16);break;
        case 'D': tocheck|=2;SBDMA=strtol(P,0,16);break;
        case 'I': tocheck|=4;SBIRQ=strtol(P,0,16);break;
        case 'H': tocheck|=8;SBDMA16=strtol(P,0,16);break;
      }

  if((tocheck&7)!=7)
  {
   printf(" Incomplete or missing BLASTER environment variable.\n");
   return 0;
  }

  if(!(tocheck&8))
   format=0;
  
  PICMask[0]=inp(0x21);
  PICMask[1]=inp(0xA1);

  if(SBIRQ>7)
   outp(0xA1,PICMask[0]|(1<<(SBIRQ-8))); /* Disable IRQ2    */
  else
   outp(0x21,PICMask[1]|(1<<SBIRQ)); /* Disable IRQ    */

  /* Reset DSP */
  outp(SBPort+0x06,0x01); /* Set the reset flag   */
  delay(10);             /* Wait for 10ms         */
  outp(SBPort+0x06,0x00); /* Clear the reset flag */
  delay(10);             /* Wait for 10ms         */


  {
   uint8 s;

   /* Check for READY status */
   if(!ReadDSP(&s))
   {
    KillSB();
    return(0);
   }
   if(s!=0xAA)
   {
    KillSB();
    return(0);
   }
   printf(" Getting DSP version...");
   DSPV=0;
   if(!WriteDSP(0xE1))
   {
    KillSB();
    return(0);
   }
   if(!ReadDSP(&s))
   {
    KillSB();
    return(0);
   }
   DSPV|=s<<8;
   if(!ReadDSP(&s))
   {
    KillSB();
    return(0);
   }
   DSPV|=s;
  }

  printf(" DSP version %d.%d...\n",DSPV>>8,DSPV&0xFF);
  if(DSPV<0x201)
  {
   printf(" DSP version number is too low.\n");
   KillSB();
   return(0);
  }

  if(DSPV<0x400)
   format=0;

  if(format)
   SBDMA=SBDMA16;
  else if(SBDMA>4)
  {
   printf("DMA out of range.");
   KillSB();
   return(0);
  }
  /* IRQ# = 0,1,2,3,4,5,6,7, DMA# = 0,1,2,3 */

  if((SBDMA==4) || (SBDMA>7))
  {
   printf("DMA out of range.\n");
   KillSB();
   return(0);
  }

  /* Allocate low memory for DMA buffer */
  J=(((fragsize*4*2*2)<<format)+15)>>4;
  if((Segment=__dpmi_allocate_dos_memory(J,&dosmemselector))==-1)
  {
   printf("Error allocating low memory.\n");
   Segment=0;
   KillSB();
   return(0);
  }

  Bluffer=Segment<<4;

  if(format)   /* Check for and fix 128KB page boundary crossing. */
  {
   if((Bluffer&0x20000) != ((Bluffer+fragsize*2*2-1)&0x20000))
    Bluffer+=fragsize*2*2;
  }
  else   /* Check for and fix 64KB page boundary crossing. */
  {
   if((Bluffer&0x10000) != ((Bluffer+fragsize*2-1)&0x10000))
    Bluffer+=fragsize*2;
  }

  DOSMemSet(Bluffer, 128, (fragsize*2)<<format);

  if(SBIRQ>7) VirtIRQ=SBIRQ+0x68;
  else VirtIRQ=SBIRQ+8;

  Pa.pm_offset=Pa2.pm_offset=(int)IRQHandler;
  Pa.pm_selector=Pa2.pm_selector=_my_cs();

  /* Set real mode interrupt vector.  */

  _go32_dpmi_get_real_mode_interrupt_vector(VirtIRQ,&PaBack2);
  _go32_dpmi_allocate_real_mode_callback_iret(&Pa2, &Pa2regs);
  _go32_dpmi_set_real_mode_interrupt_vector(VirtIRQ,&Pa2);  

  /* Set protected mode interrupt vector. */

  _go32_dpmi_get_protected_mode_interrupt_vector(VirtIRQ,&PaBack);
  _go32_dpmi_allocate_iret_wrapper(&Pa);
  _go32_dpmi_set_protected_mode_interrupt_vector(VirtIRQ,&Pa); 
  // Don't change "set" to "chain".  Bad stuff will happen.

  IRQOld=1;

  /* Enable the SoundBlaster IRQ */
  if(SBIRQ>7)
  {
   if(PICMask[0]&4)
   {
    // Disable other IRQs on PIC1 if PIC1 was disabled.
    outp(0xA1,~(1<<(SBIRQ-8)));
   }
   else
    outp(0xA1,PICMask[1]&~(1<<(SBIRQ-8)));
   outp(0x21,PICMask[0]&~4);
  }
  else
  {
   outp(0x21,PICMask[0]&~(1<<SBIRQ));
  }



  /*                       */
  /* Begin DMA programming */
  /*                       */


  if(SBDMA>3)
  {
   outp(0xd4,(SBDMA&3)|0x04);
   outp(0xd8,0x00);
   outp(0xd6,(SBDMA&3)|0x58);
  }
  else
  {
   outp(0x0A,SBDMA|0x04);
   outp(0x0C,0x00);
   outp(0x0B,SBDMA|0x58);
  }




  /*            Size of buffer */

  if(SBDMA>3)
   J=((SBDMA&3)*4)+0xc2;
  else
   J=(SBDMA*2)+1;

  outp(J,(fragsize*2-1)&0xFF);
  outp(J,(fragsize*2-1)>>8);


  /*            Page of buffer   */
  outp(PagePort[SBDMA],(Bluffer>>16)&0xFF);

  /*            Offset of buffer */
  if(SBDMA>3)
   J=((SBDMA&3)*4)+0xc0;
  else
   J=(SBDMA*2);

  outp(J,(Bluffer>>format)&0xFF);
  outp(J,(Bluffer>>(8+format))&0xFF);

  /*                       */
  /* End DMA programming   */
  /*                       */

  if(DSPV>=0x400)
  {
   WriteDSP(0x41);                // Set sampling rate
   WriteDSP(Rate>>8);             // High byte
   WriteDSP(Rate&0xFF);           // Low byte
   if(!format)
   {
    WriteDSP(0xC6);                // 8-bit output
    WriteDSP(0x00);                // 8-bit mono unsigned PCM
   }
   else
   {
    WriteDSP(0xB6);                // 16-bit output
    WriteDSP(0x10);                // 16-bit mono signed PCM
   }
   WriteDSP((fragsize-1)&0xFF);// Low byte of size
   WriteDSP((fragsize-1)>>8);  // High byte of size
  }
  else
  {
   int tc,command;

   if(Rate>22050)
   {
    tc=(65536-(256000000/Rate))>>8;
    Rate=256000000/(65536-(tc<<8));
    command=0x90;                  // High-speed auto-initialize DMA mode transfer
   }
   else
   {
    tc=256-(1000000/Rate);
    Rate=1000000/(256-tc);
    command=0x1c;                  // Auto-initialize DMA mode transfer
   }

   WriteDSP(0x40);       // Set DSP time constant
   WriteDSP(tc);         // time constant
   WriteDSP(0x48);       // Set DSP block transfer size
   WriteDSP((fragsize-1)&0xFF);
   WriteDSP((fragsize-1)>>8);

   WriteDSP(command);
  }

  if(SBDMA>3)
   outp(0xD4,SBDMA&3);          // Reenable DMA
  else
   outp(0x0A,SBDMA);            // Reenable DMA

  WriteDSP(0xD1);                 // Turn on DAC speaker
  printf(" %d hz, %d-bit\n",Rate,8<<format);
  return(Rate);
}

extern volatile int soundjoyer;
extern volatile int soundjoyeron;

static void IRQHandler(_go32_dpmi_registers *r)
{
	static volatile int Busy=0;
        uint32 *src;
	uint32 dest;
	int32 x;

        {
         uint8 status;

         outportb(SBPort+4,0x82);
         status=inportb(SBPort+5);
         if(status&1)
          inportb(SBPort+0x0E);
         else if(status&2)
          inportb(SBPort+0x0F);
         else
          return;               // Mysterious interrupt source!  *eerie music*
        }
         

        if(Busy)
        {
         if(SBIRQ>=8)
          outportb(0xA0,0x20);
         else
          outportb(0x20,0x20);

         return;
        }
        Busy=1;

        {
         uint32 count;
	 uint32 block;
	 uint32 port;

         if(SBDMA>3)
          port=((SBDMA&3)*4)+0xc2;
         else
          port=(SBDMA*2)+1;

         count=inportb(port);
         count|=inportb(port)<<8;

         if(count>=fragsize)
          block=1;
         else
          block=0;
         dest=Bluffer+((block*fragsize)<<format);
        }

        _farsetsel(_dos_ds);

        src=(uint32 *)(WaveBuffer+(ReadPtr<<format));

        for(x=(fragsize<<format)>>2;x;x--,dest+=4,src++)
        {
         _farnspokel(dest,*src);
        }

        ReadPtr=(ReadPtr+fragsize)&(fragtotal-1);

        if(soundjoyeron)
        {
         static int coot=0;
         if(!coot)
         {
          UpdateJoyData();
          soundjoyer=1;
         }
         coot=(coot+1)&3;
        }
        Busy=0;
        if(SBIRQ>=8)
         outportb(0xA0,0x20);
        else
         outportb(0x20,0x20);


}

void WriteSBSound(int32 *Buffer, int Count, int Check, int NoBlocking)
{
 int x;

 if(!format)
 {
  if(Check)
  {
   for(x=0;x<Count;x++)
   {
    int32 mix;
    while(WritePtr==ReadPtr)
     if(NoBlocking)
     {
      for(;x<Count;x++)
       Buffer[x]=0;
      return;
     }
    mix=Buffer[x]>>7;
    if(mix>127) mix=127;
    if(mix<-128) mix=-128;
    
    WaveBuffer[WritePtr]=(uint8)(mix^128);
    Buffer[x]=0;
    WritePtr=(WritePtr+1)&(fragtotal-1);
   }
  }
  else
  {
   for(x=0;x<Count;x++)
   {
    while(WritePtr==ReadPtr)
     if(NoBlocking)
     {
      for(;x<Count;x++)
       Buffer[x]=0;
      return;
     }
    WaveBuffer[WritePtr]=(uint8)((Buffer[x])>>7)^128;
    Buffer[x]=0;
    WritePtr=(WritePtr+1)&(fragtotal-1);
   }
  }
 }	
 else // 16 bit
 {
  if(Check)
  {
   for(x=0;x<Count;x++)
   {
    int32 mix;

    while(WritePtr==ReadPtr)
     if(NoBlocking)
     {
      for(;x<Count;x++)
       Buffer[x]=0;
      return;
     }
    mix=Buffer[x];
    if(mix>32767) mix=32767;
    if(mix<-32768) mix=-32768;
    ((int16 *)WaveBuffer)[WritePtr]=mix;
    Buffer[x]=0;
    WritePtr=(WritePtr+1)&(fragtotal-1);
   }
  }
  else
  {
   for(x=0;x<Count;x++)
   {
    while(WritePtr==ReadPtr)
     if(NoBlocking)
     {
      for(;x<Count;x++)
       Buffer[x]=0;
      return;
     }
    ((int16 *)WaveBuffer)[WritePtr]=Buffer[x];
    Buffer[x]=0;
    WritePtr=(WritePtr+1)&(fragtotal-1);
   }
  }
 }
}

void KillSB(void)
{
  if(SBIRQ>7)
   outp(0xA1,inp(0xA1)|(1<<(SBIRQ-8))); // Disable IRQ
  else
   outp(0x21,inp(0x21)|(1<<SBIRQ));     // Disable IRQ

  WriteDSP(0xD3);                  // Turn off speaker.
  WriteDSP(0xD0);                  // Pause DMA mode
  if(SBDMA>3)
   outp(0xd4,(SBDMA&3)|0x04);
  else
   outp(0x0A,SBDMA|0x04);           // Disable DMA

  /* If SB interrupt vector changed, restore it */
  if(IRQOld)
  {
   _go32_dpmi_set_protected_mode_interrupt_vector(VirtIRQ,&PaBack);
   _go32_dpmi_free_iret_wrapper(&Pa);
   _go32_dpmi_set_real_mode_interrupt_vector(VirtIRQ,&PaBack2);
   _go32_dpmi_free_real_mode_callback(&Pa2);
   IRQOld=0;
  }

   if(PICMask[1]>=0)
    outp(0xA1,PICMask[1]);      // PIC1
   if(PICMask[0]>=0)
    outp(0x21,PICMask[0]); 	// PIC0

  /* If low memory was allocated for DMA, free it */
  if(Segment>0)
   __dpmi_free_dos_memory(dosmemselector);

  if(WaveBuffer)
   free(WaveBuffer);
}

