/** M6502: portable 6502 emulator ****************************/
/**                                                         **/
/**                         M6502.c                         **/
/**                                                         **/
/** This file contains implementation for 6502 CPU.         **/
/**                                                         **/
/** Copyright (C) Marat Fayzullin 1996                      **/
/**               Alex Krasivsky  1996                      **/
/** Modified      BERO            1998                      **/
/** Modified	  Xodnizel	  2000-????		    **/
/**     You are not allowed to distribute this software     **/
/**     commercially. Please, notify me, if you make any    **/
/**     changes to this file.                               **/
/*************************************************************/

#include <string.h>
#include <stdio.h>
#include "m6502.h"
#include "tables.h"
#include "fce.h"
#include "sound.h"
#include "svga.h"

M6502 M;

uint32 timestamp=0;

//#define ADD_CYCLE(x) {_ICounta+=x;_ICount-=x+x+x;timestamp+=x;}

static INLINE void ADD_CYCLE(int x)
{
 M.ICounta+=x;
 M.ICount-=x+x+x;
 timestamp+=x;
}

static INLINE uint8 Op6502(uint32 A)
{
 return (M.DB=ARead[A](A));
}

#define	RdRAM(A)	(M.DB=RAM[A])
#define RdSTRAM(A)	(M.DB=(RAM+0x100)[A])

#define WrRAM(A,V)      {RAM[A]=V;}
#define WrSTRAM(A,V)	{(RAM+0x100)[A]=V;}

/** FAST_RDOP ************************************************/
/** With this #define not present, Rd6502() should perform  **/
/** the functions of Rd6502().                              **/
/*************************************************************/

#define Rd6502(A) M.DB=ARead[A](A)
#define Wr6502(A,V) BWrite[A](A,V)

#define	C_SFT	0
#define	Z_SFT	1
#define	I_SFT	2
#define	D_SFT	3
#define	B_SFT	4
#define	R_SFT	5
#define	V_SFT	6
#define	N_SFT	7


#define _A      M.A
#define _P      M.P
#define _PI	M.PI
#define _X      M.X
#define _Y      M.Y
#define _S      M.S


#define _PC     	M.PC
#define _PC_    	M.PC.W
#define _ICounta	M.ICounta
#define _ICount         M.ICount

#define jammed		M.jammed
#define	ZP	0

/** Addressing Methods ***************************************/
/** These macros calculate and return effective addresses.  **/
/*************************************************************/
#define MCZp()	(Op6502(_PC_++))
#define MCZx()	(uint8)(Op6502(_PC_++)+_X)
#define MCZy()	(uint8)(Op6502(_PC_++)+_Y)

#define MC_Ab(Rg)	M_LDWORD(Rg)
#define MC32_Ab(Rg)	M32_LDWORD(Rg)

#define MC_Ax(Rg)	M32_LDWORD(Rg);Rg+=_X;Rg&=0xFFFF
#define MC_Ay(Rg)	M32_LDWORD(Rg);Rg+=_Y;Rg&=0xFFFF

#define MC_Ix(Rg)       {uint8 IC=Op6502(_PC_)+_X;_PC_++;Rg=RdRAM(IC);Rg|=RdRAM(++IC)<<8;}
#define MC_Iy(Rg)       {uint8 IC=Op6502(_PC_);_PC_++;Rg=RdRAM(IC);Rg|=RdRAM(++IC)<<8;Rg+=_Y;Rg&=0xFFFF;}

#define MC_IyA(Rg)      {uint8 IC=Op6502(_PC_);_PC_++;Rg.B.l=RdRAM(IC);Rg.B.h=RdRAM(IC+1);if((Rg.B.l+_Y)>0xFF) {ADD_CYCLE(1);}Rg.W+=_Y;}

#define MC_AxA(Rg) {uint32 K; M32_LDWORD(Rg);K=Rg;Rg+=_X;Rg&=0xFFFF;ADD_CYCLE(((K^Rg)>>8)&1);}
#define MC_AyA(Rg) {uint32 K; M32_LDWORD(Rg);K=Rg;Rg+=_Y;Rg&=0xFFFF;ADD_CYCLE(((K^Rg)>>8)&1);}

/** Reading From Memory **************************************/
/** These macros calculate address and read from it.        **/
/*************************************************************/

#define MR_Ab(Rg)	{uint32 J;MC32_Ab(J);Rg=Rd6502(J);}
#define MR_Im(Rg)	Rg=Op6502(_PC_++)
#define	MR_Zp(Rg)	Rg=RdRAM(MCZp())
#define MR_Zx(Rg)	Rg=RdRAM(MCZx())
#define MR_Zy(Rg)	Rg=RdRAM(MCZy())

#define	MR_Ax(Rg)	{uint32 J;MC_AxA(J);Rg=Rd6502(J);}	// Modded to add 1
#define MR_Ay(Rg)	{uint32 J;MC_AyA(J);Rg=Rd6502(J);}	// if p bound crossed


#define MR_Ix(Rg)	{uint32 J;MC_Ix(J);Rg=Rd6502(J);}
#define MR_Iy(Rg)	{pair J;MC_IyA(J);Rg=Rd6502(J.W);}

/** Writing To Memory ****************************************/
/** These macros calculate address and write to it.         **/
/*************************************************************/
#define MW_Ab(Rg)	{pair J;MC_Ab(J);Wr6502(J.W,Rg);}
#define MW_Zp(Rg)	WrRAM(MCZp(),Rg)
#define MW_Zx(Rg)	WrRAM(MCZx(),Rg)
#define MW_Zy(Rg)	WrRAM(MCZy(),Rg)
#define MW_Ax(Rg)	{uint32 J;MC_Ax(J);Wr6502(J,Rg);}
#define MW_Ay(Rg)	{uint32 J;MC_Ay(J);Wr6502(J,Rg);}
#define MW_Ix(Rg)	{uint32 J;MC_Ix(J);Wr6502(J,Rg);}
#define MW_Iy(Rg)	{uint32 J;MC_Iy(J);Wr6502(J,Rg);}

/** Modifying Memory *****************************************/
/** These macros calculate address and modify it.           **/
/**						            **/
/** These macros have been modified to emulate RMW ops      **/
/** a little better(ly).				    **/
/**  -Xodnizel						    **/
/*************************************************************/
#define MM_Ab(Cmd)	{pair J;MC_Ab(J);I=Rd6502(J.W);Wr6502(J.W,I);Cmd(I);Wr6502(J.W,I);}
#define MM_Zp(Cmd)	{unsigned A=MCZp();I=RdRAM(A);Cmd(I);WrRAM(A,I); }
#define MM_Zx(Cmd)	{unsigned A=MCZx();I=RdRAM(A);Cmd(I);WrRAM(A,I); }
#define MM_Ax(Cmd)	{uint32 J;MC_Ax(J);I=Rd6502(J);Wr6502(J,I);Cmd(I);Wr6502(J,I);}
#define MM_Ay(Cmd)	{uint32 J;MC_Ay(J);I=Rd6502(J);Wr6502(J,I);Cmd(I);Wr6502(J,I);}

#define MM_Ix(Cmd)	{uint32 J;MC_Ix(J);I=Rd6502(J);Wr6502(J,I);Cmd(I);Wr6502(J,I);}
#define MM_Iy(Cmd)	{uint32 J;MC_Iy(J);I=Rd6502(J);Wr6502(J,I);Cmd(I);Wr6502(J,I);}

/** Other Macros *********************************************/
/** Calculating flags, stack, jumps, arithmetics, etc.      **/
/*************************************************************/
#define M_FL(Rg)        _P=(_P&~(Z_FLAG|N_FLAG))|ZNTable[Rg]
#define M_LDWORD(Rg)    Rg.B.l=Op6502(_PC_);_PC_++;Rg.B.h=Op6502(_PC_);_PC_++
#define M32_LDWORD(Rg)	Rg=Op6502(_PC_);_PC_++;Rg|=Op6502(_PC_)<<8;_PC_++

#define M_PUSH(Rg)	WrSTRAM(_S,Rg);_S--
#define M_POP(Rg)	_S++;Rg=RdSTRAM(_S)

#define M_JR            {uint32 KO; ADD_CYCLE(1);KO=_PC_; KO++; _PC_+=(signed char)Op6502(_PC_)+1;ADD_CYCLE((((KO^_PC_)>>8)&1));}
                                          

#define M_ADC(Rg) {uint32 K; K=_A+Rg+(_P&C_FLAG);_P&=~(N_FLAG|V_FLAG|Z_FLAG|C_FLAG);_P|=((~(_A^Rg)&(_A^K)&0x80)?V_FLAG:0)|((K&0xFF00)? C_FLAG:0)|ZNTable[K&0xFF];_A=K;}


/* Warning! C_FLAG is inverted before SBC and after it */
#define M_SBC(Rg) {pair K; K.W=_A-Rg-(~_P&C_FLAG); _P&=~(N_FLAG|V_FLAG|Z_FLAG|C_FLAG); _P|=(((_A^Rg)&(_A^K.B.l)&0x80)?V_FLAG:0)|(K.B.h? 0:C_FLAG)|ZNTable[K.B.l]; _A=K.B.l; }


#define M_CMP(Rg1,Rg2) {pair K; K.W=Rg1-Rg2;_P&=~(N_FLAG|Z_FLAG|C_FLAG);_P|=ZNTable[K.B.l]|(K.B.h? 0:C_FLAG);}
#define M_BIT(Rg) _P&=~(N_FLAG|V_FLAG|Z_FLAG);_P|=(Rg&(N_FLAG|V_FLAG))|(Rg&_A? 0:Z_FLAG);

#define M_AND(Rg)	_A&=Rg;M_FL(_A)
#define M_ANDS(Rg)	_A&=_S;M_FL(_A);
#define M_ORA(Rg)	_A|=Rg;M_FL(_A)
#define M_EOR(Rg)	_A^=Rg;M_FL(_A)
#define M_INC(Rg)	Rg++;M_FL(Rg)
#define M_DEC(Rg)	Rg--;M_FL(Rg)
#define M_ASL(Rg)	_P&=~C_FLAG;_P|=Rg>>7;Rg<<=1;M_FL(Rg)
#define M_LSR(Rg)	_P&=~C_FLAG;_P|=Rg&C_FLAG;Rg>>=1;M_FL(Rg)
#define M_ROL(Rg)	{uint8 IC=(Rg<<1)|(_P&C_FLAG);_P&=~C_FLAG;_P|=Rg>>7;Rg=IC; M_FL(Rg);}
#define M_ROR(Rg)       {uint8 IC=(Rg>>1)|(_P<<7);_P&=~C_FLAG;_P|=Rg&C_FLAG;Rg=IC; M_FL(Rg);}


#define M_DEC2(Rg)      Rg--;_P&=~C_FLAG;if(Rg==0xFF) _P|=C_FLAG;


void Add6502Cycle(int x)
{
 ADD_CYCLE(x);
}

void Reset6502(void)
{
  _PC.B.l=Op6502(0xFFFC);
  _PC.B.h=Op6502(0xFFFD);   
  if(GameAttributes&GA_NSF) _PC_=0x3812;
  jammed=0;
}

void Power6502(void)
{
  _A=_X=_Y=0x00;
  _PI=_P=Z_FLAG|R_FLAG|B_FLAG|I_FLAG;
  _S=0xFF;
  _ICounta=0;
  M.IRQlow=0;
  Reset6502();
}

void TriggerIRQ(void)
{
 M.IRQlow|=FCEU_IQTEMP;
}

static void TriggerIRQReal(void)
{
 if(!(_PI&I_FLAG) && !jammed)
 {
  ADD_CYCLE(7);
  M_PUSH(_PC.B.h);
  M_PUSH(_PC.B.l);
  M_PUSH(_P&~B_FLAG);
  _P&=~D_FLAG;
  _P|=I_FLAG;
  _PC.B.l=Op6502(0xFFFE);
  _PC.B.h=Op6502(0xFFFF);
  }
}

void FASTAPASS(1) M6502_IRQBegin(int w)
{
 M.IRQlow|=w;
}

void FASTAPASS(1) M6502_IRQEnd(int w)
{
 M.IRQlow&=~w;
}

void TriggerNMINSF(void)
{
 ADD_CYCLE(7);
 M_PUSH(_PC.B.h);
 M_PUSH(_PC.B.l);
 M_PUSH(_P&~B_FLAG);
 _P&=~D_FLAG;
 _PC_=0x3800;
}

void GINLINE TriggerNMI(void)
{
 if(!jammed)
 {
  ADD_CYCLE(7);
  M_PUSH(_PC.B.h);
  M_PUSH(_PC.B.l);
  M_PUSH(_P&~B_FLAG);
  _P&=~D_FLAG;

  _PC.B.l=Op6502(0xFFFA);
  _PC.B.h=Op6502(0xFFFB);
 }
}

/** Run6502() ************************************************/
/** This function will run 6502 code.                       **/
/*************************************************************/

void FP_FASTAPASS(1) (*MapIRQHook)(int a);
void Run6502(unsigned int cycles)
{
  register pair K;
  register uint8 I;
  int32 temp;

  M.ICount+=cycles;

  if(M.ICount<=0) return;

  M6502Loop:

    if(M.IRQlow)
    {  
     TriggerIRQReal();
     M.IRQlow&=~FCEU_IQTEMP;
     if(M.ICount<=0) {_PI=_P;return;}	/* Should increase accuracy without a major speed hit. */
    }
    _PI=_P;
    I=Op6502(_PC_);
    ADD_CYCLE(CycTable[I]);
    temp=_ICounta;
    _ICounta=0;

    if(MapIRQHook) MapIRQHook(temp);

    temp=(temp<<1)+temp;

    fhcnt-=temp;
    if(fhcnt<=0)
    {
     FrameSoundUpdate();
     fhcnt+=fhinc; //7457*3;
    }
    if(PCMIRQCount>0)
    {
     PCMIRQCount-=temp;
     if(PCMIRQCount<=0)
     {
      vdis=1;
      if((PSG[0x10]&0x80) && !(PSG[0x10]&0x40))
      {
       extern uint8 SIRQStat;
       SIRQStat|=0x80;
       M6502_IRQBegin(FCEU_IQDPCM);
      }
     }
    }

    _PC_++;
    {
     switch(I)
     {
      #include "codes.h"
     }
    }

  if(_ICount>0)
   goto M6502Loop;
}
